var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
LIBRARY({
    name: "BlockEngine",
    version: 13,
    shared: true,
    api: "CoreEngine"
});
var BlockEngine;
(function (BlockEngine) {
    var gameVersion = getMCPEVersion().array;
    /**
     * @returns game version as array
     */
    function getGameVersion() {
        return gameVersion;
    }
    BlockEngine.getGameVersion = getGameVersion;
    /**
     * @returns main game version number
     */
    function getMainGameVersion() {
        return gameVersion[1];
    }
    BlockEngine.getMainGameVersion = getMainGameVersion;
    /**
     * Sends packet with message which will be translated by the client.
     * @deprecated Use sendMessage instead.
     */
    function sendUnlocalizedMessage(client) {
        var texts = [];
        for (var _i = 1; _i < arguments.length; _i++) {
            texts[_i - 1] = arguments[_i];
        }
        client.send("blockengine.clientMessageOld", { texts: texts });
    }
    BlockEngine.sendUnlocalizedMessage = sendUnlocalizedMessage;
    function sendMessage(client, text) {
        var params = [];
        for (var _i = 2; _i < arguments.length; _i++) {
            params[_i - 2] = arguments[_i];
        }
        if (text[0] == '§' && text.length == 2 && params.length > 0) {
            var message = params.shift();
            client.send("blockengine.clientMessage", { msg: message, color: text, params: params });
        }
        else {
            client.send("blockengine.clientMessage", { msg: text, params: params });
        }
    }
    BlockEngine.sendMessage = sendMessage;
})(BlockEngine || (BlockEngine = {}));
Network.addClientPacket("blockengine.clientMessageOld", function (data) {
    var message = data.texts.map(Translation.translate).join("");
    Game.message(message);
});
Network.addClientPacket("blockengine.clientMessage", function (data) {
    var message = (data.color || "") + Translation.translate(data.msg);
    data.params.forEach(function (substr) {
        message = message.replace("%s", Translation.translate(substr));
    });
    Game.message(message);
});
var BlockEngine;
(function (BlockEngine) {
    var Decorators;
    (function (Decorators) {
        /** Client side method decorator for TileEntity */
        function ClientSide(target, propertyName) {
            target.__clientMethods = __assign({}, target.__clientMethods);
            target.__clientMethods[propertyName] = true;
        }
        Decorators.ClientSide = ClientSide;
        /** Adds method as network event in TileEntity */
        function NetworkEvent(side) {
            return function (target, propertyName) {
                target.__networkEvents = __assign({}, target.__networkEvents);
                target.__networkEvents[propertyName] = side;
            };
        }
        Decorators.NetworkEvent = NetworkEvent;
        /** Adds method as container event in TileEntity */
        function ContainerEvent(side) {
            return function (target, propertyName) {
                target.__containerEvents = __assign({}, target.__containerEvents);
                target.__containerEvents[propertyName] = side;
            };
        }
        Decorators.ContainerEvent = ContainerEvent;
    })(Decorators = BlockEngine.Decorators || (BlockEngine.Decorators = {}));
})(BlockEngine || (BlockEngine = {}));
var Side;
(function (Side) {
    Side[Side["Client"] = 0] = "Client";
    Side[Side["Server"] = 1] = "Server";
})(Side || (Side = {}));
var ItemCategory;
(function (ItemCategory) {
    ItemCategory[ItemCategory["BUILDING"] = 1] = "BUILDING";
    ItemCategory[ItemCategory["NATURE"] = 2] = "NATURE";
    ItemCategory[ItemCategory["EQUIPMENT"] = 3] = "EQUIPMENT";
    ItemCategory[ItemCategory["ITEMS"] = 4] = "ITEMS";
})(ItemCategory || (ItemCategory = {}));
var EnumRarity;
(function (EnumRarity) {
    EnumRarity[EnumRarity["COMMON"] = 0] = "COMMON";
    EnumRarity[EnumRarity["UNCOMMON"] = 1] = "UNCOMMON";
    EnumRarity[EnumRarity["RARE"] = 2] = "RARE";
    EnumRarity[EnumRarity["EPIC"] = 3] = "EPIC";
})(EnumRarity || (EnumRarity = {}));
var MiningLevel;
(function (MiningLevel) {
    MiningLevel[MiningLevel["STONE"] = 1] = "STONE";
    MiningLevel[MiningLevel["IRON"] = 2] = "IRON";
    MiningLevel[MiningLevel["DIAMOND"] = 3] = "DIAMOND";
    MiningLevel[MiningLevel["OBSIDIAN"] = 4] = "OBSIDIAN";
})(MiningLevel || (MiningLevel = {}));
/**
 * Class which represents three-dimensional vector
 * and basic operations with it.
 */
var Vector3 = /** @class */ (function () {
    function Vector3(vx, vy, vz) {
        if (typeof (vx) == "number") {
            this.x = vx;
            this.y = vy;
            this.z = vz;
        }
        else {
            var v = vx;
            this.x = v.x;
            this.y = v.y;
            this.z = v.z;
        }
    }
    /**
     * @param side block side
     * @returns direction vector for specified side
     */
    Vector3.getDirection = function (side) {
        switch (side) {
            case 0: return this.DOWN;
            case 1: return this.UP;
            case 2: return this.NORTH;
            case 3: return this.SOUTH;
            case 4: return this.EAST;
            case 5: return this.WEST;
            default: Logger.Log("Invalid block side: " + side, "ERROR");
        }
    };
    Vector3.prototype.copy = function (dst) {
        if (dst) {
            return dst.set(this);
        }
        return new Vector3(this);
    };
    Vector3.prototype.set = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            this.x = vx;
            this.y = vy;
            this.z = vz;
            return this;
        }
        var v = vx;
        return this.set(v.x, v.y, v.z);
    };
    Vector3.prototype.add = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            this.x += vx;
            this.y += vy;
            this.z += vz;
            return this;
        }
        var v = vx;
        return this.add(v.x, v.y, v.z);
    };
    /**
     * Adds vector scaled by factor.
     * @param vector vector to add.
     * @param scale scale factor
     * @returns result vector.
     */
    Vector3.prototype.addScaled = function (vector, scale) {
        return this.add(vector.x * scale, vector.y * scale, vector.z * scale);
    };
    Vector3.prototype.sub = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            this.x -= vx;
            this.y -= vy;
            this.z -= vz;
            return this;
        }
        var v = vx;
        return this.sub(v.x, v.y, v.z);
    };
    Vector3.prototype.cross = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            return this.set(this.y * vz - this.z * vy, this.z * vx - this.x * vz, this.x * vy - this.y * vx);
        }
        var v = vx;
        return this.cross(v.x, v.y, v.z);
    };
    Vector3.prototype.dot = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            return this.x * vx + this.y * vy + this.z * vz;
        }
        var v = vx;
        return this.dot(v.x, v.y, v.z);
    };
    /**
     * Normalizes vector.
     * @returns normalized vector.
     */
    Vector3.prototype.normalize = function () {
        var len = this.length();
        this.x /= len;
        this.y /= len;
        this.z /= len;
        return this;
    };
    /**
     * @returns vector length squared
     */
    Vector3.prototype.lengthSquared = function () {
        return this.x * this.x + this.y * this.y + this.z * this.z;
    };
    /**
     * @returns vector length.
     */
    Vector3.prototype.length = function () {
        return Math.sqrt(this.lengthSquared());
    };
    /**
     * Multiplies vector coords by -1.
     * @returns opposite vector.
     */
    Vector3.prototype.negate = function () {
        this.x = -this.x;
        this.y = -this.y;
        this.z = -this.z;
        return this;
    };
    Vector3.prototype.distanceSquared = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            var dx = vx - this.x;
            var dy = vy - this.y;
            var dz = vz - this.z;
            return dx * dx + dy * dy + dz * dz;
        }
        var v = vx;
        return this.distanceSquared(v.x, v.y, v.z);
    };
    Vector3.prototype.distance = function (vx, vy, vz) {
        if (typeof (vx) == "number") {
            return Math.sqrt(this.distanceSquared(vx, vy, vz));
        }
        var v = vx;
        return this.distance(v.x, v.y, v.z);
    };
    /**
     * Scales vector coords by factor.
     * @param factor scaling factor
     * @returns scaled vector
     */
    Vector3.prototype.scale = function (factor) {
        this.x *= factor;
        this.y *= factor;
        this.z *= factor;
        return this;
    };
    /**
     * Scales vector length to specified value.
     * @param len target length
     * @returns scaled vector
     */
    Vector3.prototype.scaleTo = function (len) {
        var factor = len / this.length();
        return this.scale(factor);
    };
    Vector3.prototype.toString = function () {
        return "[ " + this.x + ", " + this.y + ", " + this.z + " ]";
    };
    Vector3.DOWN = new Vector3(0, -1, 0);
    Vector3.UP = new Vector3(0, 1, 0);
    Vector3.NORTH = new Vector3(0, 0, -1);
    Vector3.SOUTH = new Vector3(0, 0, 1);
    Vector3.EAST = new Vector3(-1, 0, 0);
    Vector3.WEST = new Vector3(1, 0, 0);
    return Vector3;
}());
EXPORT("Vector3", Vector3);
/**
 * Class to work with world based on `BlockSource`
 */
var WorldRegion = /** @class */ (function () {
    function WorldRegion(blockSource) {
        this.blockSource = blockSource;
        this.isDeprecated = BlockEngine.getMainGameVersion() < 16;
    }
    /**
     * @returns interface to given dimension
     * (null if given dimension is not loaded and this interface
     * was not created yet).
     */
    WorldRegion.getForDimension = function (dimension) {
        var blockSource = BlockSource.getDefaultForDimension(dimension);
        if (blockSource) {
            return new WorldRegion(blockSource);
        }
        return null;
    };
    /**
     * @returns interface to the dimension where the given entity is
     * (null if given entity does not exist or the dimension is not loaded
     * and interface was not created).
     */
    WorldRegion.getForActor = function (entityUid) {
        var blockSource = BlockSource.getDefaultForActor(entityUid);
        if (blockSource) {
            return new WorldRegion(blockSource);
        }
        return null;
    };
    /**
     * @returns `WorldRegion` for world generation callback.
     */
    WorldRegion.getCurrentWorldGenRegion = function () {
        var blockSource = BlockSource.getCurrentWorldGenRegion();
        if (blockSource) {
            return new WorldRegion(blockSource);
        }
        return null;
    };
    /**
     * @returns the dimension id to which the following object belongs.
     */
    WorldRegion.prototype.getDimension = function () {
        return this.blockSource.getDimension();
    };
    WorldRegion.prototype.getBlock = function (x, y, z) {
        if (typeof x === "number") {
            return this.blockSource.getBlock(x, y, z);
        }
        var pos = x;
        return this.blockSource.getBlock(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.getBlockId = function (x, y, z) {
        return this.getBlock(x, y, z).id;
    };
    WorldRegion.prototype.getBlockData = function (x, y, z) {
        return this.getBlock(x, y, z).data;
    };
    WorldRegion.prototype.setBlock = function (x, y, z, id, data) {
        if (typeof x === "number") {
            if (typeof id == "number") {
                this.blockSource.setBlock(x, y, z, id, data || 0);
            }
            else {
                this.blockSource.setBlock(x, y, z, id);
            }
        }
        else {
            var pos = x;
            if (typeof arguments[1] == "number") {
                this.blockSource.setBlock(pos.x, pos.y, pos.z, arguments[1], arguments[2] || 0);
            }
            else {
                this.blockSource.setBlock(pos.x, pos.y, pos.z, arguments[1]);
            }
        }
    };
    WorldRegion.prototype.getExtraBlock = function (x, y, z) {
        if (this.isDeprecated) {
            return { id: 0, data: 0 };
        }
        if (typeof x === "number") {
            return this.blockSource.getExtraBlock(x, y, z);
        }
        var pos = x;
        return this.blockSource.getExtraBlock(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.setExtraBlock = function (x, y, z, id, data) {
        if (this.isDeprecated)
            return;
        if (typeof x === "number") {
            if (typeof id == "number") {
                this.blockSource.setExtraBlock(x, y, z, id, data || 0);
            }
            else {
                this.blockSource.setExtraBlock(x, y, z, id);
            }
        }
        else {
            var pos = x;
            if (typeof arguments[1] == "number") {
                this.blockSource.setExtraBlock(pos.x, pos.y, pos.z, arguments[1], arguments[2] || 0);
            }
            else {
                this.blockSource.setExtraBlock(pos.x, pos.y, pos.z, arguments[1]);
            }
        }
    };
    WorldRegion.prototype.destroyBlock = function (x, y, z, drop, player) {
        if (typeof x === "object") {
            var pos = x;
            return this.destroyBlock(pos.x, pos.y, pos.z, arguments[1], arguments[2]);
        }
        if (drop) {
            var block = this.getBlock(x, y, z);
            var item = player ? Entity.getCarriedItem(player) : new ItemStack();
            var result = Block.getBlockDropViaItem(block, item, new Vector3(x, y, z), this.blockSource);
            if (result) {
                for (var _i = 0, result_1 = result; _i < result_1.length; _i++) {
                    var dropItem = result_1[_i];
                    this.dropItem(x + .5, y + .5, z + .5, dropItem[0], dropItem[1], dropItem[2], dropItem[3] || null);
                }
            }
            this.blockSource.destroyBlock(x, y, z, !result);
        }
        else {
            this.blockSource.destroyBlock(x, y, z, false);
        }
    };
    WorldRegion.prototype.breakBlock = function (x, y, z, allowDrop, entity, item) {
        if (this.isDeprecated) {
            this.destroyBlock(x, y, z, allowDrop, entity);
        }
        else if (typeof x === "number") {
            this.blockSource.breakBlock(x, y, z, allowDrop, entity, item);
        }
        else {
            var pos = x;
            this.blockSource.breakBlock(pos.x, pos.y, pos.z, arguments[1], arguments[2], arguments[3]);
        }
    };
    WorldRegion.prototype.breakBlockForResult = function (x, y, z, entity, item) {
        if (typeof x === "object") {
            var pos = x;
            return this.breakBlockForResult(pos.x, pos.y, pos.z, arguments[1], arguments[2]);
        }
        if (this.isDeprecated) {
            var block = this.blockSource.getBlock(x, y, z);
            this.blockSource.setBlock(x, y, z, 0, 0);
            var level = ToolAPI.getToolLevelViaBlock(item.id, block.id);
            var drop = BlockRegistry.getBlockDrop(x, y, z, block, level, item, this.blockSource);
            var items = [];
            if (drop) {
                for (var _i = 0, drop_1 = drop; _i < drop_1.length; _i++) {
                    var item_1 = drop_1[_i];
                    items.push(new ItemStack(item_1[0], item_1[1], item_1[2], item_1[3]));
                }
            }
            return { items: items, experience: 0 };
        }
        return this.blockSource.breakBlockForJsResult(x, y, z, entity, item);
    };
    WorldRegion.prototype.getNativeTileEntity = function (x, y, z) {
        if (typeof x === "number") {
            return this.blockSource.getBlockEntity(x, y, z);
        }
        var pos = x;
        return this.blockSource.getBlockEntity(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.getTileEntity = function (x, y, z) {
        if (typeof x === "number") {
            var tileEntity = TileEntity.getTileEntity(x, y, z, this.blockSource);
            return (tileEntity && tileEntity.__initialized) ? tileEntity : null;
        }
        var pos = x;
        return this.getTileEntity(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.addTileEntity = function (x, y, z) {
        if (typeof x === "number") {
            return TileEntity.addTileEntity(x, y, z, this.blockSource);
        }
        var pos = x;
        return this.addTileEntity(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.removeTileEntity = function (x, y, z) {
        if (typeof x === "number") {
            return TileEntity.destroyTileEntityAtCoords(x, y, z, this.blockSource);
        }
        var pos = x;
        return this.removeTileEntity(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.getContainer = function (x, y, z) {
        if (typeof x === "number") {
            return World.getContainer(x, y, z, this.blockSource);
        }
        var pos = x;
        return this.getContainer(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.explode = function (x, y, z, power, fire) {
        if (typeof x === "number") {
            this.blockSource.explode(x, y, z, power, fire || false);
        }
        else {
            var pos = x;
            this.blockSource.explode(pos.x, pos.y, pos.z, arguments[1], arguments[2] || false);
        }
    };
    /**
     * @returns biome id at X and Z coord.
     */
    WorldRegion.prototype.getBiome = function (x, z) {
        return this.blockSource.getBiome(x, z);
    };
    /**
     * Sets biome id by coords.
     * @param biomeID - id of the biome to set
     */
    WorldRegion.prototype.setBiome = function (x, z, biomeID) {
        this.blockSource.setBiome(x, z, biomeID);
    };
    WorldRegion.prototype.getBiomeTemperatureAt = function (x, y, z) {
        if (typeof x === "number") {
            return this.blockSource.getBiomeTemperatureAt(x, y, z);
        }
        var pos = x;
        return this.blockSource.getBiomeTemperatureAt(pos.x, pos.y, pos.z);
    };
    /**
     * @param chunkX X coord of the chunk
     * @param chunkZ Z coord of the chunk
     * @returns true if chunk is loaded, false otherwise.
     */
    WorldRegion.prototype.isChunkLoaded = function (chunkX, chunkZ) {
        return this.blockSource.isChunkLoaded(chunkX, chunkZ);
    };
    /**
     * @param x X coord of the position
     * @param z Z coord of the position
     * @returns true if chunk on the position is loaded, false otherwise.
     */
    WorldRegion.prototype.isChunkLoadedAt = function (x, z) {
        return this.blockSource.isChunkLoadedAt(x, z);
    };
    /**
     * @param chunkX X coord of the chunk
     * @param chunkZ Z coord of the chunk
     * @returns the loading state of the chunk by chunk coords.
     */
    WorldRegion.prototype.getChunkState = function (chunkX, chunkZ) {
        return this.blockSource.getChunkState(chunkX, chunkZ);
    };
    /**
     * @param x X coord of the position
     * @param z Z coord of the position
     * @returns the loading state of the chunk by coords.
     */
    WorldRegion.prototype.getChunkStateAt = function (x, z) {
        return this.blockSource.getChunkStateAt(x, z);
    };
    WorldRegion.prototype.getLightLevel = function (x, y, z) {
        if (typeof x === "number") {
            return this.blockSource.getLightLevel(x, y, z);
        }
        var pos = x;
        return this.blockSource.getLightLevel(pos.x, pos.y, pos.z);
    };
    WorldRegion.prototype.canSeeSky = function (x, y, z) {
        if (typeof x === "number") {
            return this.blockSource.canSeeSky(x, y, z);
        }
        var pos = x;
        return this.blockSource.canSeeSky(pos.x, pos.y, pos.z);
    };
    /**
     * @returns grass color on coords
     */
    WorldRegion.prototype.getGrassColor = function (x, z) {
        return this.blockSource.getGrassColor(x, z);
    };
    WorldRegion.prototype.dropItem = function (x, y, z, id, count, data, extra) {
        if (typeof x == "object") {
            var pos = x;
            if (typeof y == "object") {
                var item = y;
                return this.dropItem(pos.x, pos.y, pos.z, item);
            }
            return this.dropItem(pos.x, pos.y, pos.z, arguments[1], arguments[2], arguments[3], arguments[4]);
        }
        if (typeof id == "object") {
            var item = id;
            return this.dropItem(x, y, z, item.id, item.count, item.data, item.extra);
        }
        return this.blockSource.spawnDroppedItem(x, y, z, id, count || 1, data || 0, extra || null);
    };
    WorldRegion.prototype.dropAtBlock = function (x, y, z, id, count, data, extra) {
        if (typeof x == "object") {
            var pos = x;
            return this.dropItem(pos.x + .5, pos.y + .5, pos.z + .5, arguments[1], arguments[2], arguments[3], arguments[4]);
        }
        return this.dropItem(x + .5, y + .5, z + .5, id, count, data, extra);
    };
    WorldRegion.prototype.spawnEntity = function (x, y, z, namespace, type, spawnEvent) {
        if (type === undefined) {
            return this.blockSource.spawnEntity(x, y, z, namespace);
        }
        return this.blockSource.spawnEntity(x, y, z, namespace, type, spawnEvent);
    };
    WorldRegion.prototype.spawnExpOrbs = function (x, y, z, amount) {
        if (typeof x == "object") {
            var pos = x;
            this.spawnExpOrbs(pos.x, pos.y, pos.z, arguments[1]);
        }
        this.blockSource.spawnExpOrbs(x, y, z, amount);
    };
    WorldRegion.prototype.listEntitiesInAABB = function (x1, y1, z1, x2, y2, z2, type, blacklist) {
        if (type === void 0) { type = -1; }
        if (blacklist === void 0) { blacklist = true; }
        if (typeof x1 == "object") {
            var pos1 = x1, pos2 = y1;
            return this.listEntitiesInAABB(pos1.x, pos1.y, pos1.z, pos2.x, pos2.y, pos2.z, z1, x2);
        }
        var entities = this.blockSource.listEntitiesInAABB(x1, y1, z1, x2, y2, z2, type, blacklist);
        if (this.isDeprecated && (type == Native.EntityType.PLAYER) != blacklist) {
            var players = Network.getConnectedPlayers();
            var dimension = this.getDimension();
            for (var _i = 0, _a = players; _i < _a.length; _i++) {
                var ent = _a[_i];
                if (Entity.getDimension(ent) != dimension)
                    continue;
                var c = Entity.getPosition(ent);
                if ((c.x >= x1 && c.x <= x2) && (c.y - 1.62 >= y1 && c.y - 1.62 <= y2) && (c.z >= z1 && c.z <= z2)) {
                    entities.push(ent);
                }
            }
        }
        return entities;
    };
    WorldRegion.prototype.playSound = function (x, y, z, name, volume, pitch, playerUids) {
        if (typeof (x) == "number") {
            if (this.blockSource.playSound) {
                this.blockSource.playSound(x, y, z, name, volume, pitch, playerUids);
            }
            else {
                var packetData = { x: x, y: y, z: z, name: name, volume: volume !== null && volume !== void 0 ? volume : 1, pitch: pitch !== null && pitch !== void 0 ? pitch : 1 };
                if (playerUids) {
                    this.sendPacketToPlayers(playerUids, "WorldRegion.play_sound", packetData);
                }
                else {
                    var radius = (volume > 1) ? 16 * volume : 16;
                    this.sendPacketInRadius(packetData, radius, "WorldRegion.play_sound", packetData);
                }
            }
        }
        else {
            var coords = arguments[0];
            this.playSound(coords.x, coords.y, coords.z, arguments[1], arguments[2], arguments[3], arguments[4]);
        }
    };
    /**
     * Plays standard Minecraft sound from the specified entity.
     * @param ent entity id
     * @param name sound name
     * @param volume sound volume from 0 to 1. Default is 1.
     * @param pitch sound pitch, from 0 to 1. Default is 1.
     * @param playerUids if not set, players in radius multiplied by sound volume
     * will be detected automatically
     */
    WorldRegion.prototype.playSoundAtEntity = function (ent, name, volume, pitch, playerUids) {
        if (volume === void 0) { volume = 1; }
        if (pitch === void 0) { pitch = 1; }
        if (this.blockSource.playSoundAtEntity) {
            this.blockSource.playSoundAtEntity(ent, name, volume, pitch);
        }
        else {
            var soundPos = Entity.getPosition(ent);
            var packetData = { ent: ent, name: name, volume: volume, pitch: pitch };
            if (playerUids) {
                this.sendPacketToPlayers(playerUids, "WorldRegion.play_sound_at", packetData);
            }
            else {
                var radius = (volume > 1) ? 16 * volume : 16;
                this.sendPacketInRadius(soundPos, radius, "WorldRegion.play_sound_at", packetData);
            }
        }
    };
    /**
     * Method to stop sound by name for defined player list.
     * @param sound resource pack sound name
     * @param playerUids list of player UIDs, if not set,
     * action will be performed on all players within dimension
     * @since Inner Core 3.1.1b127
     */
    WorldRegion.prototype.stopSound = function (sound, playerUids) {
        this.blockSource.stopSound(sound, playerUids);
    };
    /**
     * Sends network packet for players within a radius from specified coords.
     * @param coords coordinates from which players will be searched
     * @param radius radius within which players will receive packet
     * @param packetName name of the packet to send
     * @param data packet data object
     */
    WorldRegion.prototype.sendPacketInRadius = function (coords, radius, packetName, data) {
        var dimension = this.getDimension();
        var clientsList = Network.getConnectedClients();
        for (var _i = 0, _a = clientsList; _i < _a.length; _i++) {
            var client = _a[_i];
            var player = client.getPlayerUid();
            var entPos = Entity.getPosition(player);
            if (Entity.getDimension(player) == dimension && Entity.getDistanceBetweenCoords(entPos, coords) <= radius) {
                client.send(packetName, data);
            }
        }
    };
    /**
     * Sends network packet to specified players.
     * @param playerUids
     * @param packetName
     * @param data
     */
    WorldRegion.prototype.sendPacketToPlayers = function (playerUids, packetName, data) {
        for (var _i = 0, playerUids_1 = playerUids; _i < playerUids_1.length; _i++) {
            var uid = playerUids_1[_i];
            var client = Network.getClientForPlayer(uid);
            if (client) {
                client.send(packetName, data);
            }
        }
    };
    /**
     * Gets signal strength at specified coordinates
     * that consumers can receive.
     * @since Inner Core 3.1.0b125
     */
    WorldRegion.prototype.getRedstoneSignal = function (x, y, z) {
        return this.blockSource.getRedstoneSignal(x, y, z);
    };
    /**
     * Sets signal with specified strength to block, it is
     * recommended to call {@link Block.setupAsRedstoneEmitter}
     * to be able to add a source. Once block is destroyed,
     * signal will be reset.
     * @param strength level between 0-15 (inclusive)
     * @param delay time in ticks after which signal strength
     * will be reset, should be more than zero, updated depending
     * on redstone tick (1 redstone tick = 2 regular ticks), default is `4`
     * @param facing world side of {@link EBlockSide} to which signal
     * from source will be applied, use -1 to apply it to all sides
     * (as from redstone block), default is `-1`
     * @since Inner Core 3.1.0b125
     */
    WorldRegion.prototype.setRedstoneSignal = function (x, y, z, strength, delay, facing) {
        this.blockSource.setRedstoneSignal(x, y, z, strength, delay, facing);
    };
    /**
     * Causes a random tick event, usually affecting rate of
     * plant growth or grass spread and leaf disappearings.
     * @since Inner Core 3.1.0b125
     */
    WorldRegion.prototype.randomTick = function (x, y, z) {
        this.blockSource.randomTick(x, y, z);
    };
    return WorldRegion;
}());
Network.addClientPacket("WorldRegion.play_sound", function (data) {
    World.playSound(data.x, data.y, data.z, data.name, data.volume, data.pitch);
});
Network.addClientPacket("WorldRegion.play_sound_at", function (data) {
    World.playSoundAtEntity(data.ent, data.name, data.volume, data.pitch);
});
/**
 * Class to manipulate player based on `PlayerActor`.
 * Due to limitations of underlying PlayerActor class this class
 * can be used only during 1 server tick!
 */
var PlayerEntity = /** @class */ (function () {
    /**
     * Creates new instance of `PlayerEntity`.
     * @param playerUid player's numeric entity id
     */
    function PlayerEntity(playerUid) {
        this.actor = new PlayerActor(playerUid);
        this.playerUid = playerUid;
    }
    /**
     * @returns player's unique numeric entity id.
     */
    PlayerEntity.prototype.getUid = function () {
        return this.playerUid;
    };
    /**
     * @returns the id of dimension where player is.
     */
    PlayerEntity.prototype.getDimension = function () {
        return this.actor.getDimension();
    };
    /**
     * @returns player's gamemode.
     */
    PlayerEntity.prototype.getGameMode = function () {
        return this.actor.getGameMode();
    };
    PlayerEntity.prototype.addItemToInventory = function (id, count, data, extra) {
        var item = id;
        if (typeof item == "object") {
            this.actor.addItemToInventory(item.id, item.count, item.data, item.extra || null, true);
        }
        else {
            this.actor.addItemToInventory(id, count, data, extra || null, true);
        }
    };
    /**
     * @returns inventory slot's contents.
     */
    PlayerEntity.prototype.getInventorySlot = function (slot) {
        var item = this.actor.getInventorySlot(slot);
        return new ItemStack(item);
    };
    PlayerEntity.prototype.setInventorySlot = function (slot, item, count, data, extra) {
        if (extra === void 0) { extra = null; }
        if (typeof item == "object") {
            this.actor.setInventorySlot(slot, item.id, item.count, item.data, item.extra || null);
        }
        else {
            this.actor.setInventorySlot(slot, item, count, data, extra);
        }
    };
    /**
     * @returns item in player's hand
    */
    PlayerEntity.prototype.getCarriedItem = function () {
        var item = Entity.getCarriedItem(this.getUid());
        return new ItemStack(item);
    };
    PlayerEntity.prototype.setCarriedItem = function (item, count, data, extra) {
        if (extra === void 0) { extra = null; }
        if (typeof item == "object") {
            Entity.setCarriedItem(this.getUid(), item.id, item.count, item.data, item.extra);
        }
        else {
            Entity.setCarriedItem(this.getUid(), item, count, data, extra);
        }
    };
    /**
     * Decreases carried item count by specified number.
     * @param amount amount of items to decrease, default is 1
     */
    PlayerEntity.prototype.decreaseCarriedItem = function (amount) {
        if (amount === void 0) { amount = 1; }
        var item = this.getCarriedItem();
        this.setCarriedItem(item.id, item.count - amount, item.data, item.extra);
    };
    /**
     * @returns armor slot's contents.
     */
    PlayerEntity.prototype.getArmor = function (slot) {
        var item = this.actor.getArmor(slot);
        return new ItemStack(item);
    };
    PlayerEntity.prototype.setArmor = function (slot, item, count, data, extra) {
        if (extra === void 0) { extra = null; }
        if (typeof item == "object") {
            this.actor.setArmor(slot, item.id, item.count, item.data, item.extra || null);
        }
        else {
            this.actor.setArmor(slot, item, count, data, extra);
        }
    };
    /**
     * Sets respawn coords for the player.
     */
    PlayerEntity.prototype.setRespawnCoords = function (x, y, z) {
        this.actor.setRespawnCoords(x, y, z);
    };
    /**
     * Spawns exp on coords.
     * @param value experience points value
     */
    PlayerEntity.prototype.spawnExpOrbs = function (x, y, z, value) {
        this.actor.spawnExpOrbs(x, y, z, value);
    };
    /**
     * @returns whether the player is a valid entity.
     */
    PlayerEntity.prototype.isValid = function () {
        return this.actor.isValid();
    };
    /**
     * @returns player's selected slot.
     */
    PlayerEntity.prototype.getSelectedSlot = function () {
        return this.actor.getSelectedSlot();
    };
    /**
     * Sets player's selected slot.
     */
    PlayerEntity.prototype.setSelectedSlot = function (slot) {
        this.actor.setSelectedSlot(slot);
    };
    /**
     * @returns player's experience.
     */
    PlayerEntity.prototype.getExperience = function () {
        return this.actor.getExperience();
    };
    /**
     * Sets player's experience.
     */
    PlayerEntity.prototype.setExperience = function (value) {
        this.actor.setExperience(value);
    };
    /**
     * Add experience to player.
     */
    PlayerEntity.prototype.addExperience = function (amount) {
        this.actor.addExperience(amount);
    };
    /**
     * @returns player's xp level.
     */
    PlayerEntity.prototype.getLevel = function () {
        return this.actor.getLevel();
    };
    /**
     * Sets player's xp level.
     */
    PlayerEntity.prototype.setLevel = function (level) {
        this.actor.setLevel(level);
    };
    /**
     * @returns player's exhaustion.
     */
    PlayerEntity.prototype.getExhaustion = function () {
        return this.actor.getExhaustion();
    };
    /**
     * Sets player's exhaustion.
     */
    PlayerEntity.prototype.setExhaustion = function (value) {
        this.actor.setExhaustion(value);
    };
    /**
     * @returns player's hunger.
     */
    PlayerEntity.prototype.getHunger = function () {
        return this.actor.getHunger();
    };
    /**
     * Sets player's hunger.
     */
    PlayerEntity.prototype.setHunger = function (value) {
        this.actor.setHunger(value);
    };
    /**
     * @returns player's saturation.
     */
    PlayerEntity.prototype.getSaturation = function () {
        return this.actor.getSaturation();
    };
    /**
     * Sets player's saturation.
     */
    PlayerEntity.prototype.setSaturation = function (value) {
        this.actor.setSaturation(value);
    };
    /**
     * @returns player's score.
     */
    PlayerEntity.prototype.getScore = function () {
        return this.actor.getScore();
    };
    /**
     * Sets player's score.
     */
    PlayerEntity.prototype.setScore = function (value) {
        this.actor.setScore(value);
    };
    PlayerEntity.prototype.getItemUseDuration = function () {
        return this.actor.getItemUseDuration();
    };
    PlayerEntity.prototype.getItemUseIntervalProgress = function () {
        return this.actor.getItemUseIntervalProgress();
    };
    PlayerEntity.prototype.getItemUseStartupProgress = function () {
        return this.actor.getItemUseStartupProgress();
    };
    return PlayerEntity;
}());
var EntityCustomData;
(function (EntityCustomData) {
    var entities = {};
    function getAll() {
        return entities;
    }
    EntityCustomData.getAll = getAll;
    function getData(entity) {
        var data = entities[entity];
        if (!data) {
            data = {};
            putData(entity, data);
        }
        return data;
    }
    EntityCustomData.getData = getData;
    function putData(entity, data) {
        entities[entity] = data;
    }
    EntityCustomData.putData = putData;
    function getField(entity, key) {
        var playerData = getData(entity);
        if (playerData) {
            return playerData[key];
        }
    }
    EntityCustomData.getField = getField;
    function putField(entity, key, value) {
        var data = getData(entity);
        data[key] = value;
    }
    EntityCustomData.putField = putField;
    Saver.addSavesScope("EntityData", function read(scope) {
        entities = scope || {};
    }, function save() {
        return entities;
    });
    Callback.addCallback("EntityRemoved", function (entity) {
        delete entities[entity];
    });
})(EntityCustomData || (EntityCustomData = {}));
/**
 * API to store temporary data about the block.
 */
var VirtualBlockData;
(function (VirtualBlockData) {
    var cacheMap = {};
    function getKey(dimension, x, y, z) {
        return "".concat(dimension, "/").concat(x, ",").concat(y, ",").concat(z);
    }
    function getBlockEntry(dimension, x, y, z) {
        return cacheMap[getKey(dimension, x, y, z)] || null;
    }
    VirtualBlockData.getBlockEntry = getBlockEntry;
    function addBlockEntry(entry, dimension, x, y, z) {
        cacheMap[getKey(dimension, x, y, z)] = entry;
    }
    VirtualBlockData.addBlockEntry = addBlockEntry;
    function removeBlockEntry(dimension, x, y, z) {
        delete cacheMap[getKey(dimension, x, y, z)];
    }
    VirtualBlockData.removeBlockEntry = removeBlockEntry;
    Callback.addCallback("LevelLeft", function () {
        cacheMap = {};
    });
    Callback.addCallback("BreakBlock", function (blockSource, coords) {
        if (Game.isActionPrevented()) {
            return;
        }
        removeBlockEntry(blockSource.getDimension(), coords.x, coords.y, coords.z);
    }, -1);
})(VirtualBlockData || (VirtualBlockData = {}));
/**
 * Module for creating block models.
 */
var BlockModeler;
(function (BlockModeler) {
    /**
     * @returns box vertexes with specified rotation
     * @param box array of box vertexes
     * @rotation block rotation
     */
    function getRotatedBoxVertexes(box, rotation) {
        switch (rotation) {
            case 0:
                return box;
            case 1:
                return [1 - box[3], box[1], 1 - box[5], 1 - box[0], box[4], 1 - box[2]]; // rotate 180°
            case 2:
                return [box[2], box[1], 1 - box[3], box[5], box[4], 1 - box[0]]; // rotate 270°
            case 3:
                return [1 - box[5], box[1], box[0], 1 - box[2], box[4], box[3]]; // rotate 90°
        }
    }
    BlockModeler.getRotatedBoxVertexes = getRotatedBoxVertexes;
    /**
     * Sets stairs render model and shape to block.
     * @param id block numeric id
     */
    function setStairsRenderModel(id) {
        var boxes = [
            [0, 0, 0, 1, 0.5, 1],
            [0.5, 0.5, 0.5, 1, 1, 1],
            [0, 0.5, 0.5, 0.5, 1, 1],
            [0.5, 0.5, 0, 1, 1, 0.5],
            [0, 0.5, 0, 0.5, 1, 0.5]
        ];
        createStairsRenderModel(id, 0, boxes);
        var newBoxes = [];
        for (var _i = 0, boxes_1 = boxes; _i < boxes_1.length; _i++) {
            var box = boxes_1[_i];
            newBoxes.push([box[0], 1 - box[4], box[2], box[3], 1 - box[1], box[5]]);
        }
        createStairsRenderModel(id, 4, newBoxes);
    }
    BlockModeler.setStairsRenderModel = setStairsRenderModel;
    /**
     * Sets hand and ui model for the block.
     * @param blockID block numeric id
     * @param model block model
     * @param data block data (0 by default)
     */
    function setInventoryModel(blockID, model, data) {
        if (data === void 0) { data = 0; }
        ItemModel.getFor(blockID, data).setHandModel(model);
        ItemModel.getFor(blockID, data).setUiModel(model);
    }
    BlockModeler.setInventoryModel = setInventoryModel;
    function createStairsRenderModel(id, startData, boxes) {
        var modelConditionData = [
            { data: 3, posR: [-1, 0], posB: [0, 1] },
            { data: 2, posR: [1, 0], posB: [0, -1] },
            { data: 0, posR: [0, 1], posB: [1, 0] },
            { data: 1, posR: [0, -1], posB: [-1, 0] }
        ];
        for (var i = 0; i < 4; i++) {
            var conditionData = modelConditionData[i];
            var data = startData + i;
            var rBlockData = conditionData.data + startData;
            var groupR = ICRender.getGroup("stairs:" + rBlockData);
            var groupL = ICRender.getGroup("stairs:" + (rBlockData ^ 1));
            var currentGroup = ICRender.getGroup("stairs:" + data);
            currentGroup.add(id, data);
            var render = new ICRender.Model();
            var shape = new ICRender.CollisionShape();
            var box0 = boxes[0];
            render.addEntry(new BlockRenderer.Model(box0[0], box0[1], box0[2], box0[3], box0[4], box0[5], id, data)); // slabe box
            shape.addEntry().addBox(box0[0], box0[1], box0[2], box0[3], box0[4], box0[5]);
            var posR = conditionData.posR; // right block
            var posB = conditionData.posB; // back block
            var posF = [posB[0] * (-1), posB[1] * (-1)]; // front block
            var conditionRight = ICRender.BLOCK(posR[0], 0, posR[1], currentGroup, false);
            var conditionLeft = ICRender.BLOCK(posR[0] * (-1), 0, posR[1] * (-1), currentGroup, false);
            var conditionBackNotR = ICRender.BLOCK(posB[0], 0, posB[1], groupR, true);
            var conditionBackNotL = ICRender.BLOCK(posB[0], 0, posB[1], groupL, true);
            var box1 = getRotatedBoxVertexes(boxes[1], i);
            var model = new BlockRenderer.Model(box1[0], box1[1], box1[2], box1[3], box1[4], box1[5], id, data);
            var condition0 = ICRender.OR(conditionBackNotR, conditionLeft);
            render.addEntry(model).setCondition(condition0);
            shape.addEntry().addBox(box1[0], box1[1], box1[2], box1[3], box1[4], box1[5]).setCondition(condition0);
            var box2 = getRotatedBoxVertexes(boxes[2], i);
            var condition1 = ICRender.OR(conditionBackNotL, conditionRight);
            model = new BlockRenderer.Model(box2[0], box2[1], box2[2], box2[3], box2[4], box2[5], id, data);
            render.addEntry(model).setCondition(condition1);
            shape.addEntry().addBox(box2[0], box2[1], box2[2], box2[3], box2[4], box2[5]).setCondition(condition1);
            var box3 = getRotatedBoxVertexes(boxes[3], i);
            model = new BlockRenderer.Model(box3[0], box3[1], box3[2], box3[3], box3[4], box3[5], id, data);
            var condition2 = ICRender.AND(conditionBackNotR, conditionBackNotL, ICRender.NOT(conditionLeft), ICRender.BLOCK(posF[0], 0, posF[1], groupL, false));
            render.addEntry(model).setCondition(condition2);
            shape.addEntry().addBox(box3[0], box3[1], box3[2], box3[3], box3[4], box3[5]).setCondition(condition2);
            var box4 = getRotatedBoxVertexes(boxes[4], i);
            model = new BlockRenderer.Model(box4[0], box4[1], box4[2], box4[3], box4[4], box4[5], id, data);
            var condition3 = ICRender.AND(conditionBackNotR, conditionBackNotL, ICRender.NOT(conditionRight), ICRender.BLOCK(posF[0], 0, posF[1], groupR, false));
            render.addEntry(model).setCondition(condition3);
            shape.addEntry().addBox(box4[0], box4[1], box4[2], box4[3], box4[4], box4[5]).setCondition(condition3);
            BlockRenderer.setStaticICRender(id, data, render);
            BlockRenderer.setCustomCollisionShape(id, data, shape);
            BlockRenderer.setCustomRaycastShape(id, data, shape);
        }
    }
})(BlockModeler || (BlockModeler = {}));
/// <reference path="../interfaces/BlockType.ts" />
/// <reference path="../interfaces/BlockBehavior.ts" />
/**
 * Base class for block
 */
var BlockBase = /** @class */ (function () {
    function BlockBase(stringID, blockType) {
        if (blockType === void 0) { blockType = {}; }
        /** Array of block variations */
        this.variations = [];
        /** Shapes of block variations */
        this.shapes = {};
        /** Flag that defines whether block for this instance was defined or not. */
        this.isDefined = false;
        /** Block mining level */
        this.miningLevel = 0;
        /** Redstone properties */
        this.redstone = { receiver: false, connectToWires: false };
        this.stringID = stringID;
        this.id = IDRegistry.genBlockID(stringID);
        if (typeof blockType == "object") {
            BlockRegistry.extendBlockType(blockType);
        }
        else {
            blockType = BlockRegistry.getBlockType(blockType);
        }
        this.blockType = blockType;
    }
    BlockBase.prototype.addVariation = function (name, texture, inCreative) {
        if (inCreative === void 0) { inCreative = false; }
        if (!Array.isArray(texture[0])) {
            texture = [texture];
        }
        this.variations.push({ name: name, texture: texture, inCreative: inCreative });
    };
    /**
     * Registers block in game.
     */
    BlockBase.prototype.createBlock = function () {
        if (this.variations.length == 0) {
            this.addVariation(this.stringID + ".name", [["__missing", 0]]);
        }
        var blockType = this.blockType ? BlockRegistry.convertBlockTypeToSpecialType(this.blockType) : null;
        var defineDataCopy = JSON.parse(JSON.stringify(this.variations));
        for (var i = 0; i < defineDataCopy.length; i++) {
            var variation = defineDataCopy[i];
            if (variation.inCreative) {
                // Use BlockEngine's addToCreative method to avoid duplicates
                ItemRegistry.addToCreative(this.id, 1, i);
                variation.inCreative = false;
            }
        }
        Block.createBlock(this.stringID, defineDataCopy, blockType);
        this.isDefined = true;
        for (var data in this.shapes) {
            var box = this.shapes[data];
            Block.setShape(this.id, box[0], box[1], box[2], box[3], box[4], box[5], parseInt(data));
        }
        if (this.redstone.receiver) {
            Block.setupAsRedstoneReceiver(this.id, this.redstone.connectToWires);
        }
        if (this.category)
            Item.setCategory(this.id, this.category);
    };
    BlockBase.prototype.setupAsRedstoneReceiver = function (connectToWires) {
        this.redstone.receiver = true;
        this.redstone.connectToWires = connectToWires;
    };
    BlockBase.prototype.getDrop = function (coords, block, level, enchant, item, region) {
        if (level >= this.miningLevel) {
            return [[block.id, 1, block.data]];
        }
        return [];
    };
    BlockBase.prototype.onBreak = function (coords, block, region) {
        if (Math.random() >= 0.25)
            return;
        var enchant = ToolAPI.getEnchantExtraData();
        var item = new ItemStack();
        var drop = this.getDrop(coords, block, 127, enchant, item, region);
        for (var _i = 0, drop_2 = drop; _i < drop_2.length; _i++) {
            var item_2 = drop_2[_i];
            region.spawnDroppedItem(coords.x + .5, coords.y + .5, coords.z + .5, item_2[0], item_2[1], item_2[2], item_2[3] || null);
        }
    };
    /**
     * Sets destroy time for the block.
     * @param destroyTime block destroy time
     */
    BlockBase.prototype.setDestroyTime = function (destroyTime) {
        this.blockType.destroyTime = destroyTime;
    };
    /**
     * Registers block material and digging level. If you are registering
     * block with 'stone' material ensure that its block type has baseBlock
     * id 1 to be correctly destroyed by pickaxes.
     * @param material material name
     * @param level block digging level
     */
    BlockBase.prototype.setBlockMaterial = function (material, level) {
        if (level === void 0) { level = 0; }
        this.blockMaterial = material;
        this.miningLevel = level;
        BlockRegistry.setBlockMaterial(this.id, material, level);
    };
    BlockBase.prototype.setShape = function (x1, y1, z1, x2, y2, z2, data) {
        if (data === void 0) { data = -1; }
        if (typeof (x1) == "object") {
            var pos1 = x1;
            var pos2 = y1;
            data = z1;
            this.shapes[data] = [pos1.x, pos1.y, pos1.z, pos2.x, pos2.y, pos2.z];
        }
        else {
            this.shapes[data] = [x1, y1, z1, x2, y2, z2];
        }
    };
    /**
     * Sets the block type of another block, which allows to inherit some of its properties.
     * @param baseBlock id of the block to inherit type
     */
    BlockBase.prototype.setBaseBlock = function (baseBlock) {
        this.blockType.baseBlock = baseBlock;
    };
    /**
     * Sets block to be transparent or opaque.
     * @param isSolid if true, sets block to be opaque.
     */
    BlockBase.prototype.setSolid = function (isSolid) {
        this.blockType.solid = isSolid;
    };
    /**
     * Sets rendering of the block faces.
     * @param renderAllFaces If true, all block faces are rendered, otherwise back faces are not
     * rendered (for optimization purposes). Default is false
     */
    BlockBase.prototype.setRenderAllFaces = function (renderAllFaces) {
        this.blockType.renderAllFaces = renderAllFaces;
    };
    /**
     * Sets render type of the block.
     * @param renderType default is 0 (full block), use other values to change block's model
     */
    BlockBase.prototype.setRenderType = function (renderType) {
        this.blockType.renderType = renderType;
    };
    /**
     * Specifies the layer that is used to render the block.
     * @param renderLayer default is 4
     */
    BlockBase.prototype.setRenderLayer = function (renderLayer) {
        this.blockType.renderLayer = renderLayer;
    };
    /**
     * Sets level of the light emitted by the block.
     * @param lightLevel value from 0 (no light) to 15
     */
    BlockBase.prototype.setLightLevel = function (lightLevel) {
        this.blockType.lightLevel = lightLevel;
    };
    /**
     * Specifies how opaque block is.
     * @param lightOpacity Value from 0 to 15 which will be substracted
     * from the light level when the light passes through the block
     */
    BlockBase.prototype.setLightOpacity = function (lightOpacity) {
        this.blockType.lightOpacity = lightOpacity;
    };
    /**
     * Specifies how block resists to the explosions.
     * @param resistance integer value, default is 3
     */
    BlockBase.prototype.setExplosionResistance = function (resistance) {
        this.blockType.explosionResistance = resistance;
    };
    /**
     * Sets block friction. It specifies how player walks on the block.
     * The higher the friction is, the more difficult it is to change speed
     * and direction.
     * @param friction float value, default is 0.6
     */
    BlockBase.prototype.setFriction = function (friction) {
        this.blockType.friction = friction;
    };
    /**
     * Specifies rendering of shadows on the block.
     * @param translucency float value from 0 (no shadows) to 1
     */
    BlockBase.prototype.setTranslucency = function (translucency) {
        this.blockType.translucency = translucency;
    };
    /**
     * Sets sound type of the block.
     * @param sound block sound type
     */
    BlockBase.prototype.setSoundType = function (sound) {
        this.blockType.sound = sound;
    };
    /**
     * Sets block color when displayed on the vanilla maps.
     * @param color map color of the block
     */
    BlockBase.prototype.setMapColor = function (color) {
        this.blockType.mapColor = color;
    };
    /**
     * Makes block use biome color when displayed on the vanilla maps.
     * @param color block color source
     */
    BlockBase.prototype.setBlockColorSource = function (colorSource) {
        this.blockType.colorSource = colorSource;
    };
    /**
     * Sets item category.
     * @param category item category, should be integer from 1 to 4.
     */
    BlockBase.prototype.setCategory = function (category) {
        this.category = category;
    };
    /**
     * Sets item rarity.
     * @param rarity one of `EnumRarity` values
     */
    BlockBase.prototype.setRarity = function (rarity) {
        ItemRegistry.setRarity(this.id, rarity);
    };
    /**
     * Registers TileEntity prototype for this block.
     * @param prototype TileEntity prototype
     */
    BlockBase.prototype.registerTileEntity = function (prototype) {
        TileEntity.registerPrototype(this.id, prototype);
    };
    return BlockBase;
}());
/// <reference path="./BlockBase.ts" />
var BlockRotative = /** @class */ (function (_super) {
    __extends(BlockRotative, _super);
    function BlockRotative(stringID, blockType, hasVerticalFacings) {
        if (hasVerticalFacings === void 0) { hasVerticalFacings = false; }
        var _this = _super.call(this, stringID, blockType) || this;
        _this.hasVerticalFacings = hasVerticalFacings;
        return _this;
    }
    BlockRotative.prototype.addVariation = function (name, texture, inCreative) {
        var textures = [
            [texture[3], texture[2], texture[0], texture[1], texture[4], texture[5]],
            [texture[2], texture[3], texture[1], texture[0], texture[5], texture[4]],
            [texture[0], texture[1], texture[3], texture[2], texture[5], texture[4]],
            [texture[0], texture[1], texture[2], texture[3], texture[4], texture[5]],
            [texture[0], texture[1], texture[4], texture[5], texture[3], texture[2]],
            [texture[0], texture[1], texture[5], texture[4], texture[2], texture[3]]
        ];
        if (!this.hasVerticalFacings) {
            textures[0] = textures[1] = textures[3];
        }
        for (var data = 0; data < 6; data++) {
            this.variations.push({ name: name, texture: textures[data], inCreative: inCreative && data == 0 });
        }
    };
    BlockRotative.prototype.createBlock = function () {
        _super.prototype.createBlock.call(this);
        if (this.hasVerticalFacings) {
            for (var i = 0; i < this.variations.length; i += 6) {
                BlockModeler.setInventoryModel(this.id, BlockRenderer.createTexturedBlock(this.variations[i + 3].texture), i);
            }
        }
    };
    BlockRotative.prototype.onPlace = function (coords, item, block, player, region) {
        var place = BlockRegistry.getPlacePosition(coords, block, region);
        if (!place)
            return;
        var rotation = BlockRegistry.getBlockRotation(player, this.hasVerticalFacings);
        var data = (item.data - item.data % 6) + rotation;
        region.setBlock(place.x, place.y, place.z, item.id, data);
        return place;
    };
    return BlockRotative;
}(BlockBase));
/// <reference path="./BlockBase.ts" />
var BlockStairs = /** @class */ (function (_super) {
    __extends(BlockStairs, _super);
    function BlockStairs(stringID, defineData, blockType) {
        var _this = _super.call(this, stringID, blockType) || this;
        _this.variations.push(defineData);
        BlockModeler.setStairsRenderModel(_this.id);
        _this.createItemModel();
        return _this;
    }
    BlockStairs.prototype.createItemModel = function () {
        var model = BlockRenderer.createModel();
        model.addBox(0, 0, 0, 1, 0.5, 1, this.id, 0);
        model.addBox(0, 0.5, 0, 1, 1, 0.5, this.id, 0);
        BlockModeler.setInventoryModel(this.id, model);
    };
    BlockStairs.prototype.onPlace = function (coords, item, block, player, region) {
        var place = BlockRegistry.getPlacePosition(coords, block, region);
        if (!place)
            return;
        var data = BlockRegistry.getBlockRotation(player) - 2;
        if (coords.side == 0 || coords.side >= 2 && coords.vec.y - coords.y >= 0.5) {
            data += 4;
        }
        region.setBlock(place.x, place.y, place.z, item.id, data);
        return place;
    };
    return BlockStairs;
}(BlockBase));
/// <reference path="./BlockBase.ts" />
var BlockSlab = /** @class */ (function (_super) {
    __extends(BlockSlab, _super);
    function BlockSlab() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    BlockSlab.prototype.setDoubleSlab = function (blockID) {
        this.doubleSlabID = blockID;
    };
    BlockSlab.prototype.createBlock = function () {
        var defineData = this.variations;
        this.variations = [];
        for (var i = 0; i < 8; i++) {
            if (i < defineData.length) {
                this.variations.push(defineData[i]);
            }
            else {
                this.addVariation(defineData[0].name, defineData[0].texture);
            }
        }
        for (var i = 0; i < defineData.length; i++) {
            this.addVariation(defineData[i].name, defineData[i].texture, false);
        }
        for (var i = 0; i < 8; i++) {
            this.setShape(0, 0, 0, 1, 0.5, 1, i);
        }
        for (var i = 8; i < 16; i++) {
            this.setShape(0, 0.5, 0, 1, 1, 1, i);
        }
        _super.prototype.createBlock.call(this);
    };
    BlockSlab.prototype.getDrop = function (coords, block, level) {
        return [[this.id, 1, block.data % 8]];
    };
    BlockSlab.prototype.onPlace = function (coords, item, block, player, blockSource) {
        var region = new WorldRegion(blockSource);
        // make double slab
        if (block.id == item.id && block.data % 8 == item.data && Math.floor(block.data / 8) == (coords.side ^ 1)) {
            region.setBlock(coords, this.doubleSlabID, item.data);
            return;
        }
        var place = coords;
        if (!World.canTileBeReplaced(block.id, block.data)) {
            place = coords.relative;
            var tile = region.getBlock(place);
            if (!World.canTileBeReplaced(tile.id, tile.data)) {
                if (tile.id == item.id && tile.data % 8 == item.data) {
                    region.setBlock(place, this.doubleSlabID, item.data);
                }
                return;
            }
            ;
        }
        if (coords.vec.y - place.y < 0.5) {
            region.setBlock(place, item.id, item.data);
        }
        else {
            region.setBlock(place, item.id, item.data + 8);
        }
    };
    return BlockSlab;
}(BlockBase));
/// <reference path="./BlockBase.ts" />
var BlockDoubleSlab = /** @class */ (function (_super) {
    __extends(BlockDoubleSlab, _super);
    function BlockDoubleSlab() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    BlockDoubleSlab.prototype.setSlab = function (blockID) {
        this.slabID = blockID;
    };
    BlockDoubleSlab.prototype.getDrop = function (coords, block, level) {
        return [[this.slabID, 1, block.data], [this.slabID, 1, block.data]];
    };
    return BlockDoubleSlab;
}(BlockBase));
/// <reference path="./interfaces/BlockType.ts" />
/// <reference path="./interfaces/BlockBehavior.ts" />
/// <reference path="./type/BlockBase.ts" />
/// <reference path="./type/BlockRotative.ts" />
/// <reference path="./type/BlockStairs.ts" />
/// <reference path="./type/BlockSlab.ts" />
/// <reference path="./type/BlockDoubleSlab.ts" />
/**
 * Module for advanced block definition.
 */
var BlockRegistry;
(function (BlockRegistry) {
    // Import native modules
    var EntityGetYaw = ModAPI.requireGlobal("Entity.getYaw");
    var EntityGetPitch = ModAPI.requireGlobal("Entity.getPitch");
    //@ts-ignore
    var NativeBlock = com.zhekasmirnov.innercore.api.NativeBlock;
    var blocks = {};
    var blockTypes = {};
    /**
     * Creates new block using specified params.
     * @param stringID string id of the block.
     * @param defineData array containing all variations of the block. Each
     * variation corresponds to block data value, data values are assigned
     * according to variations order.
     * @param blockType BlockType object or block type name, if the type was previously registered.
     */
    function createBlock(stringID, defineData, blockType) {
        var block = new BlockBase(stringID, blockType);
        for (var _i = 0, defineData_1 = defineData; _i < defineData_1.length; _i++) {
            var variation = defineData_1[_i];
            block.addVariation(variation.name, variation.texture, variation.inCreative);
        }
        registerBlock(block);
    }
    BlockRegistry.createBlock = createBlock;
    /**
     * Creates new block with horizontal or all sides rotation.
     * @param stringID string id of the block
     * @param defineData array containing all variations of the block. Supports 2 variations,
     * each occupying 6 data values for rotation.
     * @param blockType BlockType object or block type name, if the type was previously registered.
     * @param hasVerticalFacings true if the block has vertical facings, false otherwise.
     */
    function createBlockWithRotation(stringID, defineData, blockType, hasVerticalFacings) {
        var block = new BlockRotative(stringID, blockType, hasVerticalFacings);
        for (var _i = 0, defineData_2 = defineData; _i < defineData_2.length; _i++) {
            var variation = defineData_2[_i];
            block.addVariation(variation.name, variation.texture, variation.inCreative);
        }
        registerBlock(block);
    }
    BlockRegistry.createBlockWithRotation = createBlockWithRotation;
    /**
     * Creates stairs using specified params
     * @param stringID string id of the block
     * @param defineData array containing one variation of the block (for similarity with other methods).
     * @param blockType BlockType object or block type name, if the type was previously registered.
     */
    function createStairs(stringID, defineData, blockType) {
        registerBlock(new BlockStairs(stringID, defineData[0], blockType));
    }
    BlockRegistry.createStairs = createStairs;
    /**
     * Creates slabs and its double slabs using specified params
     * @param slabID string id of the
     * @param doubleSlabID string id of the double slab
     * @param defineData array containing all variations of the block. Each
     * variation corresponds to block data value, data values are assigned
     * according to variations order.
     * @param blockType BlockType object or block type name, if the type was previously registered.
     */
    function createSlabs(slabID, doubleSlabID, defineData, blockType) {
        var slab = new BlockSlab(slabID, blockType);
        slab.variations = defineData;
        var doubleSlab = new BlockDoubleSlab(doubleSlabID, blockType);
        for (var _i = 0, defineData_3 = defineData; _i < defineData_3.length; _i++) {
            var variation = defineData_3[_i];
            doubleSlab.addVariation(variation.name, variation.texture);
        }
        slab.setDoubleSlab(doubleSlab.id);
        doubleSlab.setSlab(slab.id);
        registerBlock(slab);
        registerBlock(doubleSlab);
    }
    BlockRegistry.createSlabs = createSlabs;
    /**
     * @param name block type name
     * @returns BlockType object by name
     */
    function getBlockType(name) {
        return blockTypes[name] || null;
    }
    BlockRegistry.getBlockType = getBlockType;
    /**
     * Inherits default values from type specified in "extends" property.
     * @param type BlockType object
     */
    function extendBlockType(type) {
        if (!type.extends)
            return;
        var parent = getBlockType(type.extends);
        for (var key in parent) {
            if (!(key in type)) {
                type[key] = parent[key];
            }
        }
    }
    BlockRegistry.extendBlockType = extendBlockType;
    /**
     * Registers block type in BlockEngine and as Block.SpecialType.
     * @param name block type name
     * @param type BlockType object
     * @param isNative if true doesn't create special type
     */
    function createBlockType(name, type, isNative) {
        extendBlockType(type);
        blockTypes[name] = type;
        if (!isNative) {
            Block.createSpecialType(convertBlockTypeToSpecialType(type), name);
        }
    }
    BlockRegistry.createBlockType = createBlockType;
    /**
     * Converts block type to special type.
     * @param properites BlockType object
     * @returns block special type
     */
    function convertBlockTypeToSpecialType(properites) {
        var type = {};
        for (var key in properites) {
            switch (key) {
                case "baseBlock":
                    type.base = properites[key];
                    break;
                case "renderAllFaces":
                    type.renderallfaces = properites[key];
                    break;
                case "renderType":
                    type.rendertype = properites[key];
                    break;
                case "renderLayer":
                    type.renderlayer = properites[key];
                    break;
                case "lightLevel":
                    type.lightlevel = properites[key];
                    break;
                case "lightOpacity":
                    type.lightopacity = properites[key];
                    break;
                case "explosionResistance":
                    type.explosionres = properites[key];
                    break;
                case "destroyTime":
                    type.destroytime = properites[key];
                    break;
                case "mapColor":
                    type.mapcolor = properites[key];
                    break;
                case "colorSource":
                    type.color_source = properites[key];
                    break;
                case "canContainLiquid":
                    type.can_contain_liquid = properites[key];
                    break;
                case "canBeExtraBlock":
                    type.can_be_extra_block = properites[key];
                    break;
                case "flameOdds":
                    type.flame_odds = properites[key];
                    break;
                case "burnOdds":
                    type.burn_odds = properites[key];
                    break;
                case "extends": continue;
                default:
                    type[key] = properites[key];
                    break;
            }
        }
        return type;
    }
    BlockRegistry.convertBlockTypeToSpecialType = convertBlockTypeToSpecialType;
    /**
     * @param blockID block numeric or string id
     * @returns instance of block class if it exists
     */
    function getInstanceOf(blockID) {
        var numericID = Block.getNumericId(blockID);
        return blocks[numericID] || null;
    }
    BlockRegistry.getInstanceOf = getInstanceOf;
    /**
     * Registers instance of BlockBase class and creates block for it.
     * @param block instance of BlockBase class
     * @returns the same BlockBase instance with `isDefined` flag set to true
     */
    function registerBlock(block) {
        block.createBlock();
        registerBlockFuncs(block.id, block);
        blocks[block.id] = block;
        return block;
    }
    BlockRegistry.registerBlock = registerBlock;
    /**
     * Registers all block functions.
     * @param blockID block numeric or string id
     * @param blockFuncs object containing block functions
     */
    function registerBlockFuncs(blockID, blockFuncs) {
        var numericID = Block.getNumericId(blockID);
        if ('getDrop' in blockFuncs) {
            Block.registerDropFunction(numericID, function (coords, blockID, blockData, diggingLevel, enchant, item, region) {
                return blockFuncs.getDrop(coords, { id: blockID, data: blockData }, diggingLevel, enchant, new ItemStack(item), region);
            });
        }
        if ('onDestroy' in blockFuncs) {
            Callback.addCallback("DestroyBlock", function (coords, block, player) {
                if (block.id == numericID) {
                    blockFuncs.onDestroy(coords, block, BlockSource.getDefaultForActor(player), player);
                }
            });
        }
        if ('onBreak' in blockFuncs) {
            Block.registerPopResourcesFunction(numericID, function (coords, block, region) {
                blockFuncs.onBreak(coords, block, region);
            });
        }
        if ('onPlace' in blockFuncs) {
            Block.registerPlaceFunction(numericID, function (coords, item, block, player, region) {
                return blockFuncs.onPlace(coords, new ItemStack(item), block, player, region);
            });
        }
        if ('onNeighbourChange' in blockFuncs) {
            Block.registerNeighbourChangeFunction(numericID, function (coords, block, changeCoords, region) {
                blockFuncs.onNeighbourChange(coords, block, changeCoords, region);
            });
        }
        if ('onEntityInside' in blockFuncs) {
            Block.registerEntityInsideFunction(numericID, function (coords, block, entity) {
                blockFuncs.onEntityInside(coords, block, entity);
            });
        }
        if ('onEntityStepOn' in blockFuncs) {
            Block.registerEntityStepOnFunction(numericID, function (coords, block, entity) {
                blockFuncs.onEntityStepOn(coords, block, entity);
            });
        }
        if ('onRandomTick' in blockFuncs) {
            Block.setRandomTickCallback(numericID, function (x, y, z, id, data, region) {
                blockFuncs.onRandomTick(x, y, z, { id: id, data: data }, region);
            });
        }
        if ('onAnimateTick' in blockFuncs) {
            Block.setAnimateTickCallback(numericID, function (x, y, z, id, data) {
                blockFuncs.onAnimateTick(x, y, z, id, data);
            });
        }
        if ('onClick' in blockFuncs) {
            if (Block.registerClickFunction) {
                Block.registerClickFunction(numericID, function (coords, item, block, player) {
                    blockFuncs.onClick(coords, new ItemStack(item), block, player);
                });
            }
            else {
                Callback.addCallback("ItemUse", function (coords, item, block, isExternal, player) {
                    if (block.id == numericID) {
                        blockFuncs.onClick(coords, new ItemStack(item), block, player);
                    }
                });
            }
        }
        if ('onNameOverride' in blockFuncs || 'onItemUse' in blockFuncs || 'onDispense' in blockFuncs) {
            ItemRegistry.registerItemFuncs(blockID, blockFuncs);
        }
    }
    BlockRegistry.registerBlockFuncs = registerBlockFuncs;
    /**
     * Sets destroy time for the block with specified id.
     * @param blockID block numeric or string id
     * @param time block destroy time
     */
    function setDestroyTime(blockID, time) {
        Block.setDestroyTime(blockID, time);
    }
    BlockRegistry.setDestroyTime = setDestroyTime;
    /**
     * Sets the block type of another block, which allows to inherit some of its properties.
     * @param blockID block numeric or string id
     * @param baseBlock id of the block to inherit type
     */
    function setBaseBlock(blockID, baseBlock) {
        NativeBlock.setMaterialBase(Block.getNumericId(blockID), baseBlock);
    }
    BlockRegistry.setBaseBlock = setBaseBlock;
    /**
     * Sets block to be transparent or opaque.
     * @param blockID block numeric or string id
     * @param isSolid if true, sets block to be opaque.
     */
    function setSolid(blockID, isSolid) {
        NativeBlock.setSolid(Block.getNumericId(blockID), isSolid);
    }
    BlockRegistry.setSolid = setSolid;
    /**
     * Sets rendering of the block faces.
     * @param blockID block numeric or string id
     * @param renderAllFaces If true, all block faces are rendered, otherwise back faces are not
     * rendered (for optimization purposes). Default is false
     */
    function setRenderAllFaces(blockID, renderAllFaces) {
        NativeBlock.setRenderAllFaces(Block.getNumericId(blockID), renderAllFaces);
    }
    BlockRegistry.setRenderAllFaces = setRenderAllFaces;
    /**
     * Sets render type of the block.
     * @param blockID block numeric or string id
     * @param renderType default is 0 (full block), use other values to change block's model
     */
    function setRenderType(blockID, renderType) {
        NativeBlock.setRenderType(Block.getNumericId(blockID), renderType);
    }
    BlockRegistry.setRenderType = setRenderType;
    /**
     * Specifies the layer that is used to render the block.
     * @param blockID block numeric or string id
     * @param renderLayer default is 4
     */
    function setRenderLayer(blockID, renderLayer) {
        NativeBlock.setRenderLayer(Block.getNumericId(blockID), renderLayer);
    }
    BlockRegistry.setRenderLayer = setRenderLayer;
    /**
     * Sets level of the light emitted by the block.
     * @param blockID block numeric or string id
     * @param lightLevel value from 0 (no light) to 15
     */
    function setLightLevel(blockID, lightLevel) {
        NativeBlock.setLightLevel(Block.getNumericId(blockID), lightLevel);
    }
    BlockRegistry.setLightLevel = setLightLevel;
    /**
     * Specifies how opaque block is.
     * @param blockID block numeric or string id
     * @param lightOpacity Value from 0 to 15 which will be substracted
     * from the light level when the light passes through the block
     */
    function setLightOpacity(blockID, lightOpacity) {
        NativeBlock.setLightOpacity(Block.getNumericId(blockID), lightOpacity);
    }
    BlockRegistry.setLightOpacity = setLightOpacity;
    /**
     * Specifies how block resists to the explosions.
     * @param blockID block numeric or string id
     * @param resistance integer value, default is 3
     */
    function setExplosionResistance(blockID, resistance) {
        NativeBlock.setExplosionResistance(Block.getNumericId(blockID), resistance);
    }
    BlockRegistry.setExplosionResistance = setExplosionResistance;
    /**
     * Sets block friction. It specifies how player walks on the block.
     * The higher the friction is, the more difficult it is to change speed
     * and direction.
     * @param blockID block numeric or string id
     * @param friction float value, default is 0.6
     */
    function setFriction(blockID, friction) {
        NativeBlock.setFriction(Block.getNumericId(blockID), friction);
    }
    BlockRegistry.setFriction = setFriction;
    /**
     * Specifies rendering of shadows on the block.
     * @param blockID block numeric or string id
     * @param translucency float value from 0 (no shadows) to 1
     */
    function setTranslucency(blockID, translucency) {
        NativeBlock.setTranslucency(Block.getNumericId(blockID), translucency);
    }
    BlockRegistry.setTranslucency = setTranslucency;
    /**
     * Sets sound type of the block.
     * @param blockID block numeric or string id
     * @param sound block sound type
     */
    function setSoundType(blockID, sound) {
        NativeBlock.setSoundType(Block.getNumericId(blockID), sound);
    }
    BlockRegistry.setSoundType = setSoundType;
    /**
     * Sets block color when displayed on the vanilla maps.
     * @param blockID block numeric or string id
     * @param color map color of the block
     */
    function setMapColor(blockID, color) {
        NativeBlock.setMapColor(Block.getNumericId(blockID), color);
    }
    BlockRegistry.setMapColor = setMapColor;
    /**
     * Makes block use biome color when displayed on the vanilla maps.
     * @param blockID block numeric or string id
     * @param color block color source
     */
    function setBlockColorSource(blockID, color) {
        NativeBlock.setBlockColorSource(Block.getNumericId(blockID), color);
    }
    BlockRegistry.setBlockColorSource = setBlockColorSource;
    /**
     * Registers block material and digging level. If you are registering
     * block with 'stone' material ensure that its block type has baseBlock
     * id 1 to be correctly destroyed by pickaxes.
     * @param blockID block numeric or string id
     * @param material material name
     * @param level block's digging level
     */
    function setBlockMaterial(blockID, material, level) {
        ToolAPI.registerBlockMaterial(Block.getNumericId(blockID), material, level, material == "stone");
    }
    BlockRegistry.setBlockMaterial = setBlockMaterial;
    /**
     * @returns block side opposite to player rotation.
     * @param player player uid
     * @param hasVertical if true can return vertical sides as well
     */
    function getBlockRotation(player, hasVertical) {
        var pitch = EntityGetPitch(player);
        if (hasVertical) {
            if (pitch < -45)
                return 0;
            if (pitch > 45)
                return 1;
        }
        var rotation = Math.floor((EntityGetYaw(player) - 45) % 360 / 90);
        if (rotation < 0)
            rotation += 4;
        rotation = [5, 3, 4, 2][rotation];
        return rotation;
    }
    BlockRegistry.getBlockRotation = getBlockRotation;
    /**
     * @returns block place position for click coords in world.
     * @param coords click coords
     * @param block touched block
     * @param region BlockSource
     */
    function getPlacePosition(coords, block, region) {
        if (World.canTileBeReplaced(block.id, block.data))
            return coords;
        var place = coords.relative;
        block = region.getBlock(place.x, place.y, place.z);
        if (World.canTileBeReplaced(block.id, block.data))
            return place;
        return null;
    }
    BlockRegistry.getPlacePosition = getPlacePosition;
    /**
     * Registers place function for block with rotation.
     * @param id block numeric or string id
     * @param hasVertical true if the block has vertical facings, false otherwise.
     */
    function setRotationFunction(id, hasVertical, placeSound) {
        Block.registerPlaceFunction(id, function (coords, item, block, player, region) {
            var place = getPlacePosition(coords, block, region);
            if (!place)
                return;
            var rotation = getBlockRotation(player, hasVertical);
            region.setBlock(place.x, place.y, place.z, item.id, (item.data - item.data % 6) + rotation);
            //World.playSound(place.x, place.y, place.z, placeSound || "dig.stone", 1, 0.8);
            return place;
        });
    }
    BlockRegistry.setRotationFunction = setRotationFunction;
    /**
     * Registers drop function for block.
     * @param blockID block numeric or string id
     * @param dropFunc drop function
     * @param level mining level
     */
    function registerDrop(blockID, dropFunc, level) {
        Block.registerDropFunction(blockID, function (blockCoords, blockID, blockData, diggingLevel, enchant, item, region) {
            if (!level || diggingLevel >= level) {
                return dropFunc(blockCoords, blockID, blockData, diggingLevel, enchant, item, region);
            }
            return [];
        });
        addBlockDropOnExplosion(blockID);
    }
    BlockRegistry.registerDrop = registerDrop;
    /**
     * Sets mining level for block.
     * @param blockID block numeric or string id
     * @param level mining level
     */
    function setDestroyLevel(blockID, level) {
        Block.registerDropFunction(blockID, function (coords, blockID, blockData, diggingLevel) {
            if (diggingLevel >= level) {
                return [[Block.getNumericId(blockID), 1, 0]];
            }
        });
        addBlockDropOnExplosion(blockID);
    }
    BlockRegistry.setDestroyLevel = setDestroyLevel;
    /**
     * Registers function called when block is destroyed by explosion.
     * @param blockID block numeric or string id
     * @param func function on explosion
     */
    function registerOnExplosionFunction(blockID, func) {
        Block.registerPopResourcesFunction(blockID, func);
    }
    BlockRegistry.registerOnExplosionFunction = registerOnExplosionFunction;
    /**
     * Registers block drop on explosion with 25% chance.
     */
    function addBlockDropOnExplosion(blockID) {
        Block.registerPopResourcesFunction(blockID, function (coords, block, region) {
            if (Math.random() >= 0.25)
                return;
            var dropFunc = Block.getDropFunction(block.id);
            var enchant = ToolAPI.getEnchantExtraData();
            var item = new ItemStack();
            //@ts-ignore
            var drop = dropFunc(coords, block.id, block.data, 127, enchant, item, region);
            for (var _i = 0, drop_3 = drop; _i < drop_3.length; _i++) {
                var item_3 = drop_3[_i];
                region.spawnDroppedItem(coords.x + .5, coords.y + .5, coords.z + .5, item_3[0], item_3[1], item_3[2], item_3[3] || null);
            }
        });
    }
    BlockRegistry.addBlockDropOnExplosion = addBlockDropOnExplosion;
    var noDropBlocks = [26, 30, 31, 32, 51, 59, 92, 99, 100, 104, 105, 106, 115, 127, 132, 141, 142, 144, 161, 175, 199, 244, 385, 386, 388, 389, 390, 391, 392, 462];
    /** @deprecated */
    function getBlockDrop(x, y, z, block, level, item, region) {
        var id = block.id, data = block.data;
        var enchant = ToolAPI.getEnchantExtraData(item.extra);
        //@ts-ignore
        var dropFunc = Block.dropFunctions[id];
        if (dropFunc) {
            region !== null && region !== void 0 ? region : (region = BlockSource.getDefaultForActor(Player.get()));
            return dropFunc(new Vector3(x, y, z), id, data, level, enchant, item, region);
        }
        if (id == 3 || id == 5 || id == 6 || id == 12 || id == 19 || id == 35 || id == 85 || id == 158 || id == 171)
            return [[id, 1, data]];
        if (id == 17 || id == 162)
            return [[id, 1, data]]; // log
        if (id == 18 || id == 161) { // leaves
            if (enchant.silk)
                return [[id, 1, data]];
            return [];
        }
        if (id == 47) { // bookshelf
            if (enchant.silk)
                return [[id, 1, 0]];
            return [[340, 3, 0]];
        }
        if (id == 55)
            return [[331, 1, 0]]; // redstone wire
        if (id == 60)
            return [[3, 1, 0]]; // farmland
        if (id == 63 || id == 68)
            return [[338, 1, 0]]; // sign
        if (id == 64)
            return [[324, 1, 0]]; // door
        if (id == 75 || id == 76)
            return [[76, 1, 0]]; // redstone torch
        if (id == 79) { // ice
            if (enchant.silk)
                return [[id, 1, 0]];
            return [];
        }
        if (id == 83)
            return [[338, 1, 0]]; // sugar canes
        if (id == 89)
            return [[348, Math.floor(Math.random() * 3 + 2), 0]]; // glowstone
        if (id == 93 || id == 94)
            return [[356, 1, 0]]; // repeater
        if (id == 103)
            return [[360, Math.floor(Math.random() * 4 + 4), 0]]; // melon
        if (id == 123 || id == 124)
            return [[123, 1, 0]]; // redstone lamp
        if (id == 140)
            return [[390, 1, 0]]; // pot
        if (id == 149 || id == 150)
            return [[404, 1, 0]]; // comparator
        if (id == 151 || id == 178)
            return [[151, 1, 0]]; // daylight detector
        // doors
        if (id == 193)
            return [[427, 1, 0]];
        if (id == 194)
            return [[428, 1, 0]];
        if (id == 195)
            return [[429, 1, 0]];
        if (id == 196)
            return [[430, 1, 0]];
        if (id == 197)
            return [[431, 1, 0]];
        if (id == 393)
            return [[335, 1, 0]]; // kelp
        if (id == VanillaTileID.campfire) {
            if (enchant.silk)
                return [[id, 1, 0]];
            var item_4 = IDConverter.getIDData("charcoal");
            return [[item_4.id, 1, item_4.data]];
        }
        if (id == VanillaTileID.soul_campfire) {
            if (enchant.silk)
                return [[id, 1, 0]];
            return [[VanillaTileID.soul_soil, 1, 0]];
        }
        // signs
        if (id == 436 || id == 437)
            return [[472, 1, 0]];
        if (id == 441 || id == 442)
            return [[473, 1, 0]];
        if (id == 443 || id == 444)
            return [[474, 1, 0]];
        if (id == 445 || id == 446)
            return [[475, 1, 0]];
        if (id == 447 || id == 448)
            return [[476, 1, 0]];
        if (id == 467)
            return [[-212, 1, data]]; // wood
        if (noDropBlocks.indexOf(id) != -1)
            return [];
        return [[Block.convertBlockToItemId(id), 1, 0]];
    }
    BlockRegistry.getBlockDrop = getBlockDrop;
    // default block types
    createBlockType("opaque", {
        baseBlock: 1,
        solid: true,
        lightOpacity: 15,
        explosionResistance: 4,
        renderLayer: 2,
        translucency: 0,
        sound: "stone"
    }, true);
    createBlockType("stone", {
        extends: "opaque",
        destroyTime: 1.5,
        explosionResistance: 30
    });
    createBlockType("ore", {
        extends: "opaque",
        destroyTime: 3,
        explosionResistance: 15
    });
    createBlockType("wood", {
        extends: "opaque",
        baseBlock: 17,
        destroyTime: 2,
        explosionResistance: 10,
        flameOdds: 5,
        burnOdds: 20,
        sound: "wood"
    });
    createBlockType("leaves", {
        baseBlock: 18,
        destroyTime: 0.2,
        explosionResistance: 1,
        renderAllFaces: true,
        renderLayer: 1,
        lightOpacity: 1,
        translucency: 0.5,
        flameOdds: 30,
        burnOdds: 60,
        sound: "grass"
    });
    createBlockType("dirt", {
        extends: "opaque",
        baseBlock: 2,
        destroyTime: 0.5,
        explosionResistance: 2.5,
        sound: "gravel"
    });
    Callback.addCallback("RedstoneSignal", function (coords, params, onLoad, blockSource) {
        var blockId = blockSource.getBlockId(coords.x, coords.y, coords.z);
        var instance = getInstanceOf(blockId);
        if (instance && 'onRedstoneUpdate' in instance) {
            instance.onRedstoneUpdate(coords, params, blockSource);
        }
    });
})(BlockRegistry || (BlockRegistry = {}));
/// <reference path="./BlockItemBehavior.ts" />
var ItemBase = /** @class */ (function () {
    function ItemBase(stringID, name, icon) {
        /**
         * Maximum stack size of the item
         */
        this.maxStack = 64;
        /**
         * Maximum data value of the item
         */
        this.maxDamage = 0;
        this.inCreative = false;
        this.stringID = stringID;
        this.id = IDRegistry.genItemID(stringID);
        this.setName(name || stringID);
        if (typeof icon == "string")
            this.setIcon(icon);
        else if (typeof icon == "object")
            this.setIcon(icon.name, icon.meta || icon.data);
        else
            this.setIcon("missing_icon");
    }
    /**
     * Method that can be overrided to modify item name before item creation.
     * @param name item name passed to the constructor
     */
    ItemBase.prototype.setName = function (name) {
        this.name = name;
    };
    /**
     * Method that can be overrided to modify item textures before item creation.
     * @param texture texture name
     * @param index texture index
     */
    ItemBase.prototype.setIcon = function (texture, index) {
        if (index === void 0) { index = 0; }
        this.icon = { name: texture, meta: index };
    };
    /**
     * Sets item creative category
     * @param category item category, should be integer from 1 to 4.
     */
    ItemBase.prototype.setCategory = function (category) {
        Item.setCategory(this.id, category);
    };
    /**
     * Sets item maximum stack size
     * @param maxStack maximum stack size for the item
     */
    ItemBase.prototype.setMaxStack = function (maxStack) {
        this.maxStack = maxStack;
        this.item.setMaxStackSize(maxStack);
    };
    /**
     * Sets item maximum data value
     * @param maxDamage maximum data value for the item
     */
    ItemBase.prototype.setMaxDamage = function (maxDamage) {
        this.maxDamage = maxDamage;
        this.item.setMaxDamage(maxDamage);
    };
    /**
    * Specifies how the player should hold the item
    * @param enabled if true, player holds the item as a tool, not as a simple
    * item
    */
    ItemBase.prototype.setHandEquipped = function (enabled) {
        this.item.setHandEquipped(enabled);
    };
    /**
     * Allows item to be put in off hand
     */
    ItemBase.prototype.allowInOffHand = function () {
        this.item.setAllowedInOffhand(true);
    };
    /**
     * Allows item to click on liquid blocks
     */
    ItemBase.prototype.setLiquidClip = function () {
        this.item.setLiquidClip(true);
    };
    /**
     * Specifies how the item can be enchanted
     * @param type enchant type defining whan enchants can or cannot be
     * applied to this item, one of the Native.EnchantType
     * @param enchantability quality of the enchants that are applied, the higher this
     * value is, the better enchants you get with the same level
     */
    ItemBase.prototype.setEnchantType = function (type, enchantability) {
        this.item.setEnchantType(type, enchantability);
    };
    /**
     * Sets item as glint (like enchanted tools or golden apple)
     * @param enabled if true, the item will be displayed as glint item
     */
    ItemBase.prototype.setGlint = function (enabled) {
        this.item.setGlint(enabled);
    };
    /**
     * Adds material that can be used to repair the item in the anvil
     * @param itemID item id to be used as repair material
     */
    ItemBase.prototype.addRepairItem = function (itemID) {
        this.item.addRepairItem(itemID);
    };
    /**
    * Sets properties for the item from JSON-like object. Uses vanilla mechanics.
    * @param props object containing properties
    */
    ItemBase.prototype.setProperties = function (props) {
        this.item.setProperties(JSON.stringify(props));
    };
    /**
     * Sets item rarity.
     * @param rarity one of `EnumRarity` values
     */
    ItemBase.prototype.setRarity = function (rarity) {
        ItemRegistry.setRarity(this.id, rarity);
    };
    return ItemBase;
}());
var ItemCommon = /** @class */ (function (_super) {
    __extends(ItemCommon, _super);
    function ItemCommon(stringID, name, icon, inCreative) {
        if (inCreative === void 0) { inCreative = true; }
        var _this = _super.call(this, stringID, name, icon) || this;
        _this.item = Item.createItem(_this.stringID, _this.name, _this.icon, { isTech: true });
        _this.setCategory(ItemCategory.ITEMS);
        if (inCreative)
            ItemRegistry.addToCreative(_this.id, 1, 0);
        return _this;
    }
    return ItemCommon;
}(ItemBase));
var ItemFood = /** @class */ (function (_super) {
    __extends(ItemFood, _super);
    function ItemFood(stringID, name, icon, params, inCreative) {
        if (inCreative === void 0) { inCreative = true; }
        var _this = this;
        var _a;
        _this = _super.call(this, stringID, name, icon) || this;
        var foodProperties = {
            nutrition: params.food || 0,
            saturation_modifier: params.saturation || "normal",
            is_meat: params.isMeat || false,
            can_always_eat: params.canAlwaysEat || false,
            effects: params.effects || []
        };
        if (params.usingConvertsTo) {
            foodProperties["using_converts_to"] = params.usingConvertsTo;
        }
        (_a = params.useDuration) !== null && _a !== void 0 ? _a : (params.useDuration = 32);
        if (BlockEngine.getMainGameVersion() == 11) {
            _this.setProperties({
                use_duration: params.useDuration,
                food: foodProperties
            });
        }
        else {
            _this.setProperties({
                components: {
                    "minecraft:use_duration": params.useDuration,
                    "minecraft:food": foodProperties
                }
            });
        }
        _this.item.setUseAnimation(1);
        _this.item.setMaxUseDuration(params.useDuration);
        return _this;
    }
    ItemFood.prototype.onFoodEaten = function (item, food, saturation, player) { };
    return ItemFood;
}(ItemCommon));
Callback.addCallback("FoodEaten", function (food, saturation, player) {
    var item = Entity.getCarriedItem(player);
    var itemInstance = ItemRegistry.getInstanceOf(item.id);
    if (itemInstance && itemInstance.onFoodEaten) {
        itemInstance.onFoodEaten(item, food, saturation, player);
    }
});
var ItemThrowable = /** @class */ (function (_super) {
    __extends(ItemThrowable, _super);
    function ItemThrowable(stringID, name, icon, inCreative) {
        if (inCreative === void 0) { inCreative = true; }
        var _this = _super.call(this, stringID, name, icon) || this;
        _this.item = Item.createThrowableItem(_this.stringID, _this.name, _this.icon, { isTech: true });
        _this.setCategory(ItemCategory.ITEMS);
        if (inCreative)
            ItemRegistry.addToCreative(_this.id, 1, 0);
        Item.registerThrowableFunctionForID(_this.id, function (projectile, item, target) {
            _this.onProjectileHit(projectile, item, target);
        });
        return _this;
    }
    ItemThrowable.prototype.onProjectileHit = function (projectile, item, target) { };
    return ItemThrowable;
}(ItemBase));
/// <reference path="../interfaces/ArmorListeners.ts" />
var ItemArmor = /** @class */ (function (_super) {
    __extends(ItemArmor, _super);
    function ItemArmor(stringID, name, icon, params, inCreative) {
        if (inCreative === void 0) { inCreative = true; }
        var _this = _super.call(this, stringID, name, icon) || this;
        _this.armorType = params.type;
        _this.defence = params.defence;
        _this.setArmorTexture(params.texture);
        _this.item = Item.createArmorItem(_this.stringID, _this.name, _this.icon, {
            type: _this.armorType,
            armor: _this.defence,
            knockbackResist: params.knockbackResistance * 0.1125 || 0,
            durability: 0,
            texture: _this.texture,
            isTech: true
        });
        _this.setCategory(ItemCategory.EQUIPMENT);
        if (params.material)
            _this.setMaterial(params.material);
        if (inCreative)
            ItemRegistry.addToCreative(_this.id, 1, 0);
        ItemArmor.registerListeners(_this.id, _this);
        return _this;
    }
    /**
     * Method that can be overrided to modify armor texture before item creation.
     * @param texture armor texture path
     */
    ItemArmor.prototype.setArmorTexture = function (texture) {
        this.texture = texture;
    };
    /**
     * Sets armor properties from armor material.
     * @param armorMaterial material name or object.
     */
    ItemArmor.prototype.setMaterial = function (armorMaterial) {
        if (typeof armorMaterial == "string") {
            armorMaterial = ItemRegistry.getArmorMaterial(armorMaterial);
        }
        this.armorMaterial = armorMaterial;
        var index = Native.ArmorType[this.armorType];
        var maxDamage = armorMaterial.durabilityFactor * ItemArmor.maxDamageArray[index];
        this.setMaxDamage(maxDamage);
        if (armorMaterial.enchantability) {
            this.setEnchantType(Native.EnchantType[this.armorType], armorMaterial.enchantability);
        }
        if (armorMaterial.repairMaterial) {
            this.addRepairItem(armorMaterial.repairMaterial);
        }
    };
    /**
     * Prevents armor from being damaged.
     */
    ItemArmor.prototype.preventDamaging = function () {
        Armor.preventDamaging(this.id);
    };
    /**
     * Registers all armor functions from given object.
     * @param id armor item id
     * @param armorFuncs object that implements `ArmorListener` interface
     */
    ItemArmor.registerListeners = function (id, armorFuncs) {
        if ('onHurt' in armorFuncs) {
            Armor.registerOnHurtListener(id, function (item, slot, player, type, value, attacker, bool1, bool2) {
                return armorFuncs.onHurt({ attacker: attacker, type: type, damage: value, bool1: bool1, bool2: bool2 }, item, slot, player);
            });
        }
        if ('onTick' in armorFuncs) {
            Armor.registerOnTickListener(id, function (item, slot, player) {
                return armorFuncs.onTick(item, slot, player);
            });
        }
        if ('onTakeOn' in armorFuncs) {
            Armor.registerOnTakeOnListener(id, function (item, slot, player) {
                armorFuncs.onTakeOn(item, slot, player);
            });
        }
        if ('onTakeOff' in armorFuncs) {
            Armor.registerOnTakeOffListener(id, function (item, slot, player) {
                armorFuncs.onTakeOff(item, slot, player);
            });
        }
    };
    ItemArmor.maxDamageArray = [11, 16, 15, 13];
    return ItemArmor;
}(ItemBase));
/// <reference path="../interfaces/ToolParams.ts" />
var ItemTool = /** @class */ (function (_super) {
    __extends(ItemTool, _super);
    function ItemTool(stringID, name, icon, toolMaterial, toolData, inCreative) {
        var _this = _super.call(this, stringID, name, icon, inCreative) || this;
        _this.handEquipped = false;
        _this.brokenId = 0;
        _this.damage = 1;
        _this.isWeapon = false;
        _this.blockTypes = [];
        _this.setMaxStack(1);
        _this.setCategory(ItemCategory.EQUIPMENT);
        if (typeof toolMaterial == "string") {
            toolMaterial = ItemRegistry.getToolMaterial(toolMaterial);
        }
        _this.toolMaterial = toolMaterial;
        if (toolData) {
            for (var key in toolData) {
                _this[key] = toolData[key];
            }
        }
        ToolAPI.registerTool(_this.id, _this.toolMaterial, _this.blockTypes, _this);
        if (_this.enchantType && toolMaterial.enchantability) {
            _this.setEnchantType(_this.enchantType, toolMaterial.enchantability);
        }
        if (toolMaterial.repairMaterial) {
            _this.addRepairItem(toolMaterial.repairMaterial);
        }
        if (_this.handEquipped) {
            _this.setHandEquipped(true);
        }
        return _this;
    }
    return ItemTool;
}(ItemCommon));
/// <reference path="./interfaces/ItemBehavior.ts" />
/// <reference path="./type/ItemBase.ts" />
/// <reference path="./type/ItemCommon.ts" />
/// <reference path="./type/ItemFood.ts" />
/// <reference path="./type/ItemThrowable.ts" />
/// <reference path="./type/ItemArmor.ts" />
/// <reference path="./type/ItemTool.ts" />
/**
 * Module for advanced item definition.
 */
var ItemRegistry;
(function (ItemRegistry) {
    var items = {};
    var itemsRarity = {};
    var armorMaterials = {};
    /**
     * Map items without extra data by "id:data" key.
     */
    var creativeTabItems = {};
    /**
     * @returns item type
     */
    function getType(id) {
        return IDRegistry.getIdInfo(id).split(":")[0];
    }
    ItemRegistry.getType = getType;
    /**
     * @param id block id
     * @returns true if a block identifier was given, false otherwise.
     */
    function isBlock(id) {
        return getType(id) == "block";
    }
    ItemRegistry.isBlock = isBlock;
    /**
     * @param id item id
     * @returns true if an item identifier was given, false otherwise.
     */
    function isItem(id) {
        return getType(id) == "item";
    }
    ItemRegistry.isItem = isItem;
    /**
     * @returns whether the item is an item from the original game.
     */
    function isVanilla(id) {
        return !IDRegistry.getNameByID(id);
    }
    ItemRegistry.isVanilla = isVanilla;
    /**
     * @returns item string id in the game (in snake_case format).
     */
    function getVanillaStringID(id) {
        return IDRegistry.getIdInfo(id).split(":")[1].split("#")[0];
    }
    ItemRegistry.getVanillaStringID = getVanillaStringID;
    /**
     * @returns instance of item class if the item was added by BlockEngine, null otherwise.
     */
    function getInstanceOf(itemID) {
        var numericID = Item.getNumericId(itemID);
        return items[numericID] || null;
    }
    ItemRegistry.getInstanceOf = getInstanceOf;
    /**
     * @returns `EnumRarity` value for the item.
     */
    function getRarity(itemID) {
        var _a;
        return (_a = itemsRarity[itemID]) !== null && _a !== void 0 ? _a : EnumRarity.COMMON;
    }
    ItemRegistry.getRarity = getRarity;
    /**
     * @returns chat color for rarity.
     * @param rarity one of `EnumRarity` values
     */
    function getRarityColor(rarity) {
        if (rarity == EnumRarity.UNCOMMON)
            return "§e";
        if (rarity == EnumRarity.RARE)
            return "§b";
        if (rarity == EnumRarity.EPIC)
            return "§d";
        return "";
    }
    ItemRegistry.getRarityColor = getRarityColor;
    /**
     * @returns chat color for rare items.
     */
    function getItemRarityColor(itemID) {
        return getRarityColor(getRarity(itemID));
    }
    ItemRegistry.getItemRarityColor = getItemRarityColor;
    /**
     * Sets item rarity.
     * @param id item id
     * @param rarity one of `EnumRarity` values
     * @param preventNameOverride prevent registration of name override function
     */
    function setRarity(id, rarity, preventNameOverride) {
        var numericID = Item.getNumericId(id);
        itemsRarity[numericID] = rarity;
        //@ts-ignore
        if (!preventNameOverride && !Item.nameOverrideFunctions[numericID]) {
            Item.registerNameOverrideFunction(numericID, function (item, translation, name) {
                return getItemRarityColor(item.id) + translation;
            });
        }
    }
    ItemRegistry.setRarity = setRarity;
    /**
     * Creates new armor material with specified parameters.
     * @param name new (or existing) material name
     * @param material material properties
     */
    function addArmorMaterial(name, material) {
        armorMaterials[name] = material;
    }
    ItemRegistry.addArmorMaterial = addArmorMaterial;
    /**
     * @returns armor material by name.
     */
    function getArmorMaterial(name) {
        return armorMaterials[name];
    }
    ItemRegistry.getArmorMaterial = getArmorMaterial;
    /**
     * Registers new tool material in ToolAPI. Some of the tool
     * materials are already registered:
     * *wood*, *stone*, *iron*, *golden* and *diamond*
     * @param name new (or existing) material name
     * @param material material properties
     */
    function addToolMaterial(name, material) {
        ToolAPI.addToolMaterial(name, material);
    }
    ItemRegistry.addToolMaterial = addToolMaterial;
    /**
     * @returns tool material by name registered in ToolAPI.
     */
    function getToolMaterial(name) {
        //@ts-ignore
        return ToolAPI.toolMaterials[name];
    }
    ItemRegistry.getToolMaterial = getToolMaterial;
    /**
     * Registers item instance and it's functions.
     * @param itemInstance item class instance
     * @returns item instance back
     */
    function registerItem(itemInstance) {
        items[itemInstance.id] = itemInstance;
        registerItemFuncs(itemInstance.id, itemInstance);
        return itemInstance;
    }
    ItemRegistry.registerItem = registerItem;
    /**
     * Registers all item functions from given object.
     * @param itemFuncs object which implements `ItemBehavior` interface
     */
    function registerItemFuncs(itemID, itemFuncs) {
        var numericID = Item.getNumericId(itemID);
        if ('onNameOverride' in itemFuncs) {
            Item.registerNameOverrideFunction(numericID, function (item, translation, name) {
                return getItemRarityColor(item.id) + itemFuncs.onNameOverride(item, translation, name);
            });
        }
        if ('onIconOverride' in itemFuncs) {
            Item.registerIconOverrideFunction(numericID, function (item, isModUi) {
                return itemFuncs.onIconOverride(item, isModUi);
            });
        }
        if ('onItemUse' in itemFuncs) {
            Item.registerUseFunction(numericID, function (coords, item, block, player) {
                itemFuncs.onItemUse(coords, new ItemStack(item), block, player);
            });
        }
        if ('onNoTargetUse' in itemFuncs) {
            Item.registerNoTargetUseFunction(numericID, function (item, player) {
                itemFuncs.onNoTargetUse(new ItemStack(item), player);
            });
        }
        if ('onUsingReleased' in itemFuncs) {
            Item.registerUsingReleasedFunction(numericID, function (item, ticks, player) {
                itemFuncs.onUsingReleased(new ItemStack(item), ticks, player);
            });
        }
        if ('onUsingComplete' in itemFuncs) {
            Item.registerUsingCompleteFunction(numericID, function (item, player) {
                itemFuncs.onUsingComplete(new ItemStack(item), player);
            });
        }
        if ('onDispense' in itemFuncs) {
            Item.registerDispenseFunction(numericID, function (coords, item, blockSource, slot) {
                var region = new WorldRegion(blockSource);
                itemFuncs.onDispense(coords, new ItemStack(item), region, slot);
            });
        }
    }
    ItemRegistry.registerItemFuncs = registerItemFuncs;
    /**
     * Creates item from given description. Automatically generates item id
     * from given string id.
     * @param stringID item string id.
     * @param params item description
     * @returns item class instance
     */
    function createItem(stringID, params) {
        var item;
        if (params.type == "food") {
            return createFood(stringID, params);
        }
        else if (params.type == "throwable") {
            item = new ItemThrowable(stringID, params.name, params.icon, params.inCreative);
        }
        else {
            item = new ItemCommon(stringID, params.name, params.icon, params.inCreative);
        }
        item.setCategory(params.category || ItemCategory.ITEMS);
        if (params.stack)
            item.setMaxStack(params.stack);
        if (params.maxDamage)
            item.setMaxDamage(params.maxDamage);
        if (params.handEquipped)
            item.setHandEquipped(true);
        if (params.allowedInOffhand)
            item.allowInOffHand();
        if (params.glint)
            item.setGlint(true);
        if (params.enchant)
            item.setEnchantType(params.enchant.type, params.enchant.value);
        if (params.rarity)
            item.setRarity(params.rarity);
        items[item.id] = item;
        return item;
    }
    ItemRegistry.createItem = createItem;
    function createFood(stringID, params) {
        var item = new ItemFood(stringID, params.name, params.icon, params, params.inCreative);
        if (params.stack)
            item.setMaxStack(params.stack);
        if (params.category)
            item.setCategory(params.category);
        if (params.glint)
            item.setGlint(true);
        if (params.rarity)
            item.setRarity(params.rarity);
        items[item.id] = item;
        return item;
    }
    ItemRegistry.createFood = createFood;
    ;
    /**
     * Creates armor item from given description. Automatically generates item id
     * from given string id.
     * @param stringID item string id
     * @param params item and armor parameters
     * @returns item class instance
     */
    function createArmor(stringID, params) {
        var item = new ItemArmor(stringID, params.name, params.icon, params, params.inCreative);
        registerItem(item);
        if (params.category)
            item.setCategory(params.category);
        if (params.glint)
            item.setGlint(true);
        if (params.rarity)
            item.setRarity(params.rarity);
        return item;
    }
    ItemRegistry.createArmor = createArmor;
    ;
    /**
     * Creates tool item and registers it in ToolAPI. Automatically generates item id
     * from given string id.
     * @param stringID item string id
     * @param params object with item parameters and tool material
     * @param toolData tool parameters and functions
     * @returns item class instance
     */
    function createTool(stringID, params, toolData) {
        var item = new ItemTool(stringID, params.name, params.icon, params.material, toolData, params.inCreative);
        registerItem(item);
        if (params.category)
            item.setCategory(params.category);
        if (params.glint)
            item.setGlint(true);
        if (params.rarity)
            item.setRarity(params.rarity);
        return item;
    }
    ItemRegistry.createTool = createTool;
    /**
     * Adds item to creative. If extra data is not specified and item with same id and data is already added, it will be skipped.
     * @param id item id
     * @param count item count
     * @param data item data
     * @param extra item extra data
     */
    function addToCreative(id, count, data, extra) {
        if (extra) {
            Item.addToCreative(id, count, data, extra);
        }
        else {
            var mappingKey = "".concat(id, ":").concat(data);
            if (creativeTabItems[mappingKey])
                return;
            creativeTabItems[mappingKey] = true;
            Item.addToCreative(id, count, data);
        }
    }
    ItemRegistry.addToCreative = addToCreative;
})(ItemRegistry || (ItemRegistry = {}));
/**
 * Class representing item stack in the inventory.
 */
var ItemStack = /** @class */ (function () {
    function ItemStack(item, count, data, extra) {
        if (typeof item == "object") {
            this.id = item.id;
            this.data = item.data;
            this.count = item.count;
            this.extra = item.extra || null;
        }
        else {
            this.id = item || 0;
            this.data = data || 0;
            this.count = count || 0;
            this.extra = extra || null;
        }
    }
    /**
     * @returns instance of item class if the item was added by BlockEngine, null otherwise.
     */
    ItemStack.prototype.getItemInstance = function () {
        return ItemRegistry.getInstanceOf(this.id);
    };
    /**
     * Creates a copy of current ItemStack object
     * @returns a created copy of the ItemStack
     */
    ItemStack.prototype.copy = function () {
        var _a;
        return new ItemStack(this.id, this.count, this.data, (_a = this.extra) === null || _a === void 0 ? void 0 : _a.copy());
    };
    /**
     * @returns maximum stack size for the item
     */
    ItemStack.prototype.getMaxStack = function () {
        return Item.getMaxStack(this.id);
    };
    /**
     * @returns maximum damage value for the item
     */
    ItemStack.prototype.getMaxDamage = function () {
        return Item.getMaxDamage(this.id);
    };
    /**
     * @returns true if all stack values are empty, false otherwise
     */
    ItemStack.prototype.isEmpty = function () {
        return this.id == 0 &&
            this.count == 0 &&
            this.data == 0 &&
            this.extra == null;
    };
    /**
     * Decreases stack count by specified value.
     * @param count amount to decrease
     */
    ItemStack.prototype.decrease = function (count) {
        this.count -= count;
        if (this.count <= 0)
            this.clear();
    };
    /**
     * Sets all stack values to 0.
     */
    ItemStack.prototype.clear = function () {
        this.id = this.data = this.count = 0;
        this.extra = null;
    };
    /**
     * Applies damage to the item and destroys it if its max damage reached
     * @param damage amount to apply
     */
    ItemStack.prototype.applyDamage = function (damage) {
        var unbreakingLevel = this.getEnchantLevel(Native.Enchantment.UNBREAKING);
        if (Math.random() < 1 / (unbreakingLevel + 1)) {
            this.data += damage;
        }
        if (this.data >= this.getMaxDamage()) {
            var tool = ToolAPI.getToolData(this.id);
            if (tool && tool.brokenId) {
                this.id = tool.brokenId;
                this.data = 0;
                this.extra = null;
            }
            else {
                this.clear();
            }
        }
    };
    /**
     * @returns item's custom name
     */
    ItemStack.prototype.getCustomName = function () {
        var _a;
        return ((_a = this.extra) === null || _a === void 0 ? void 0 : _a.getCustomName()) || "";
    };
    /**
    * Sets item's custom name. Creates new ItemExtraData instance if
    * it doesn't exist.
    */
    ItemStack.prototype.setCustomName = function (name) {
        var _a;
        (_a = this.extra) !== null && _a !== void 0 ? _a : (this.extra = new ItemExtraData());
        this.extra.setCustomName(name);
    };
    /**
     * @returns true if the item is enchanted, false otherwise
     */
    ItemStack.prototype.isEnchanted = function () {
        var _a;
        return ((_a = this.extra) === null || _a === void 0 ? void 0 : _a.isEnchanted()) || false;
    };
    /**
     * Adds a new enchantment to the item. Creates new ItemExtraData instance if
     * it doesn't exist.
     * @param id enchantment id, one of the Native.Enchantment constants
     * @param level enchantment level, generally between 1 and 5
     */
    ItemStack.prototype.addEnchant = function (id, level) {
        var _a;
        (_a = this.extra) !== null && _a !== void 0 ? _a : (this.extra = new ItemExtraData());
        this.extra.addEnchant(id, level);
    };
    /**
     * Removes enchantments by its id
     * @param id enchantment id, one of the Native.Enchantment constants
     */
    ItemStack.prototype.removeEnchant = function (id) {
        var _a;
        (_a = this.extra) === null || _a === void 0 ? void 0 : _a.removeEnchant(id);
    };
    /**
     * Removes all the enchantments of the item
     */
    ItemStack.prototype.removeAllEnchants = function () {
        var _a;
        (_a = this.extra) === null || _a === void 0 ? void 0 : _a.removeAllEnchants();
    };
    /**
     * @param id enchantment id, one of the Native.Enchantment constants
     * @returns level of the specified enchantment
     */
    ItemStack.prototype.getEnchantLevel = function (id) {
        var _a;
        return ((_a = this.extra) === null || _a === void 0 ? void 0 : _a.getEnchantLevel(id)) || 0;
    };
    /**
     * @returns all the enchantments of the item in the readable format
     */
    ItemStack.prototype.getEnchants = function () {
        var _a;
        return ((_a = this.extra) === null || _a === void 0 ? void 0 : _a.getEnchants()) || null;
    };
    return ItemStack;
}());
/// <reference path="./interfaces/ToolParams.ts" />
/**
 * Tool parameters for vanilla tool types.
 */
var ToolType;
(function (ToolType) {
    ToolType.SWORD = {
        __flag: "__sword",
        handEquipped: true,
        isWeapon: true,
        enchantType: Native.EnchantType.weapon,
        damage: 4,
        blockTypes: ["fibre", "plant"],
        calcDestroyTime: function (item, coords, block, params, destroyTime, enchant) {
            if (block.id == 30)
                return 0.08;
            var material = ToolAPI.getBlockMaterialName(block.id);
            if (material == "plant" || block.id == 86 || block.id == 91 || block.id == 103 || block.id == 127 || block.id == 410) {
                return params.base / 1.5;
            }
            return destroyTime;
        }
    };
    ToolType.SHOVEL = {
        __flag: "__shovel",
        handEquipped: true,
        enchantType: Native.EnchantType.shovel,
        damage: 2,
        blockTypes: ["dirt"],
        onItemUse: function (coords, item, block, player) {
            if (block.id == 2 && coords.side == 1) {
                var region = WorldRegion.getForActor(player);
                region.setBlock(coords, 198, 0);
                region.playSound(coords.x + .5, coords.y + 1, coords.z + .5, "step.grass", 0.5, 0.8);
                item.applyDamage(1);
                Entity.setCarriedItem(player, item.id, item.count, item.data, item.extra);
            }
        }
    };
    ToolType.PICKAXE = {
        __flag: "__pickaxe",
        handEquipped: true,
        enchantType: Native.EnchantType.pickaxe,
        damage: 2,
        blockTypes: ["stone"],
    };
    ToolType.AXE = {
        __flag: "__axe",
        handEquipped: true,
        enchantType: Native.EnchantType.axe,
        damage: 3,
        blockTypes: ["wood"],
        onItemUse: function (coords, item, tile, player) {
            var logID = this.getStrippedLogId(tile);
            if (logID) {
                var region = WorldRegion.getForActor(player);
                if (BlockEngine.getMainGameVersion() >= 16) {
                    var block = region.getBlock(coords);
                    var states = { "pillar_axis": block.getState(EBlockStates.PILLAR_AXIS) };
                    if (logID == VanillaTileID.stripped_warped_stem || logID == VanillaTileID.stripped_crimson_stem) {
                        states["deprecated"] = 0;
                    }
                    var block2 = new BlockState(logID, states);
                    region.setBlock(coords, block2);
                }
                else {
                    region.setBlock(coords, logID, 0);
                }
                item.applyDamage(1);
                Entity.setCarriedItem(player, item.id, item.count, item.data, item.extra);
            }
        },
        getStrippedLogId: function (block) {
            switch (block.id) {
                case 17:
                    if (block.data == 0)
                        return VanillaTileID.stripped_oak_log;
                    if (block.data == 1)
                        return VanillaTileID.stripped_spruce_log;
                    if (block.data == 2)
                        return VanillaTileID.stripped_birch_log;
                    return VanillaTileID.stripped_jungle_log;
                case 162:
                    if (block.data == 0)
                        return VanillaTileID.stripped_acacia_log;
                    return VanillaTileID.stripped_dark_oak_log;
                case VanillaTileID.warped_stem:
                    return VanillaTileID.stripped_warped_stem;
                case VanillaTileID.crimson_stem:
                    return VanillaTileID.stripped_crimson_stem;
            }
        }
    };
    ToolType.HOE = {
        __flag: "__hoe",
        handEquipped: true,
        enchantType: Native.EnchantType.pickaxe,
        damage: 2,
        blockTypes: ["plant"],
        onItemUse: function (coords, item, block, player) {
            if ((block.id == 2 || block.id == 3) && coords.side != 0) {
                var region = WorldRegion.getForActor(player);
                if (region.getBlockId(coords.x, coords.y + 1, coords.z) != 0)
                    return;
                region.setBlock(coords, 60, 0);
                region.playSound(coords.x + .5, coords.y + 1, coords.z + .5, "step.gravel", 1, 0.8);
                item.applyDamage(1);
                Entity.setCarriedItem(player, item.id, item.count, item.data, item.extra);
            }
        }
    };
    ToolType.SHEARS = {
        __flag: "__shears",
        blockTypes: ["plant", "fibre", "wool"],
        modifyEnchant: function (enchantData, item, coords, block) {
            if (block) {
                var material = ToolAPI.getBlockMaterialName(block.id);
                if (material == "fibre" || material == "plant") {
                    enchantData.silk = true;
                }
            }
        },
        calcDestroyTime: function (item, coords, block, params, destroyTime, enchant) {
            if (block.id == 30)
                return 0.08;
            return destroyTime;
        },
        onDestroy: function (item, coords, block, player) {
            if (block.id == 31 || block.id == 32 || block.id == 18 || block.id == 161) {
                var region = WorldRegion.getForActor(player);
                region.destroyBlock(coords);
                region.dropItem(coords.x + .5, coords.y + .5, coords.z + .5, block.id, 1, block.data);
            }
            return false;
        }
    };
})(ToolType || (ToolType = {}));
ToolAPI.addBlockMaterial("wool", 1.5);
ToolAPI.registerBlockMaterial(35, "wool");
// By NikolaySavenko (https://github.com/NikolaySavenko)
/**
 * Module to convert item ids depending on the game version.
 */
var IDConverter;
(function (IDConverter) {
    var oldIDPairs = {};
    /**
     * Registers old id and data for new string id.
     */
    function registerOld(stringId, oldId, oldData) {
        oldIDPairs[stringId] = { id: oldId, data: oldData };
    }
    IDConverter.registerOld = registerOld;
    /**
     * Creates ItemStack instance by string id.
     * @param stringId new item string id
     * @param count item count
     * @param data item data
     * @param extra item extra data
     * @returns ItemStack instance
     */
    function getStack(stringId, count, data, extra) {
        if (count === void 0) { count = 1; }
        if (data === void 0) { data = 0; }
        if (extra === void 0) { extra = null; }
        if (BlockEngine.getMainGameVersion() == 11) {
            var oldPair = oldIDPairs[stringId];
            if (oldPair) {
                return new ItemStack(oldPair.id, count, oldPair.data, extra);
            }
        }
        return new ItemStack(getNumericId(stringId), count, data, extra);
    }
    IDConverter.getStack = getStack;
    /**
     * @param stringId new item string id
     * @returns converted id data pair
     */
    function getIDData(stringId) {
        if (BlockEngine.getMainGameVersion() == 11 && oldIDPairs.hasOwnProperty(stringId)) {
            return oldIDPairs[stringId];
        }
        return { id: getNumericId(stringId), data: 0 };
    }
    IDConverter.getIDData = getIDData;
    /**
     * @param stringId new item string id
     * @returns converted numeric id
     */
    function getID(stringId) {
        if (BlockEngine.getMainGameVersion() == 11 && oldIDPairs.hasOwnProperty(stringId)) {
            return oldIDPairs[stringId].id;
        }
        return getNumericId(stringId);
    }
    IDConverter.getID = getID;
    /**
     * @param stringId new item string id
     * @returns converted data
     */
    function getData(stringId) {
        if (BlockEngine.getMainGameVersion() == 11 && oldIDPairs.hasOwnProperty(stringId)) {
            return oldIDPairs[stringId].data;
        }
        return 0;
    }
    IDConverter.getData = getData;
    function getNumericId(stringId) {
        return VanillaItemID[stringId] || VanillaBlockID[stringId];
    }
})(IDConverter || (IDConverter = {}));
/// <reference path="IDConverter.ts" />
IDConverter.registerOld("charcoal", 263, 1);
IDConverter.registerOld("oak_boat", 333, 0);
IDConverter.registerOld("spruce_boat", 333, 1);
IDConverter.registerOld("birch_boat", 333, 2);
IDConverter.registerOld("jungle_boat", 333, 3);
IDConverter.registerOld("acacia_boat", 333, 4);
IDConverter.registerOld("dark_oak_boat", 333, 5);
IDConverter.registerOld("milk_bucket", 325, 1);
IDConverter.registerOld("water_bucket", 325, 8);
IDConverter.registerOld("lava_bucket", 325, 10);
IDConverter.registerOld("ink_sac", 351, 0);
IDConverter.registerOld("red_dye", 351, 1);
IDConverter.registerOld("green_dye", 351, 2);
IDConverter.registerOld("cocoa_beans", 351, 3);
IDConverter.registerOld("lapis_lazuli", 351, 4);
IDConverter.registerOld("purple_dye", 351, 5);
IDConverter.registerOld("cyan_dye", 351, 6);
IDConverter.registerOld("light_gray_dye", 351, 7);
IDConverter.registerOld("gray_dye", 351, 8);
IDConverter.registerOld("pink_dye", 351, 9);
IDConverter.registerOld("lime_dye", 351, 10);
IDConverter.registerOld("yellow_dye", 351, 11);
IDConverter.registerOld("light_blue_dye", 351, 12);
IDConverter.registerOld("magenta_dye", 351, 13);
IDConverter.registerOld("orange_dye", 351, 14);
IDConverter.registerOld("bone_meal", 351, 15);
IDConverter.registerOld("black_dye", 351, 16);
IDConverter.registerOld("brown_dye", 351, 17);
IDConverter.registerOld("blue_dye", 351, 18);
IDConverter.registerOld("white_dye", 351, 19);
IDConverter.registerOld("cooked_cod", VanillaItemID.cooked_fish, 0);
IDConverter.registerOld("cod", VanillaItemID.fish, 0);
IDConverter.registerOld("tropical_fish", VanillaItemID.clownfish, 0);
IDConverter.registerOld("melon_slice", VanillaItemID.melon, 0);
IDConverter.registerOld("leather_horse_armor", VanillaItemID.horsearmorleather, 0);
IDConverter.registerOld("iron_horse_armor", VanillaItemID.horsearmoriron, 0);
IDConverter.registerOld("golden_horse_armor", VanillaItemID.horsearmorgold, 0);
IDConverter.registerOld("diamond_horse_armor", VanillaItemID.horsearmordiamond, 0);
IDConverter.registerOld("mutton", VanillaItemID.muttonraw, 0);
IDConverter.registerOld("cooked_mutton", VanillaItemID.muttoncooked, 0);
IDConverter.registerOld("totem_of_undying", VanillaItemID.totem, 0);
IDConverter.registerOld("music_disc_13", VanillaItemID.record_13, 0);
IDConverter.registerOld("music_disc_cat", VanillaItemID.record_cat, 0);
IDConverter.registerOld("music_disc_blocks", VanillaItemID.record_blocks, 0);
IDConverter.registerOld("music_disc_chirp", VanillaItemID.record_chirp, 0);
IDConverter.registerOld("music_disc_far", VanillaItemID.record_far, 0);
IDConverter.registerOld("music_disc_mall", VanillaItemID.record_mall, 0);
IDConverter.registerOld("music_disc_mellohi", VanillaItemID.record_mellohi, 0);
IDConverter.registerOld("music_disc_stal", VanillaItemID.record_stal, 0);
IDConverter.registerOld("music_disc_strad", VanillaItemID.record_strad, 0);
IDConverter.registerOld("music_disc_ward", VanillaItemID.record_ward, 0);
IDConverter.registerOld("music_disc_11", VanillaItemID.record_11, 0);
IDConverter.registerOld("music_disc_wait", VanillaItemID.record_wait, 0);
var TileEntityBase = /** @class */ (function () {
    function TileEntityBase() {
        this.useNetworkItemContainer = true;
        this._clickPrevented = false;
        this.client = {
            load: this.clientLoad,
            unload: this.clientUnload,
            tick: this.clientTick,
            events: {},
            containerEvents: {}
        };
        this.events = {};
        this.containerEvents = {};
        for (var propertyName in this.__clientMethods) {
            this.client[propertyName] = this[propertyName];
        }
        for (var eventName in this.__networkEvents) {
            var side = this.__networkEvents[eventName];
            var target = (side == Side.Client) ? this.client.events : this.events;
            target[eventName] = this[eventName];
        }
        for (var eventName in this.__containerEvents) {
            var side = this.__containerEvents[eventName];
            var target = (side == Side.Client) ? this.client.containerEvents : this.containerEvents;
            target[eventName] = this[eventName];
        }
        delete this.__clientMethods;
        delete this.__networkEvents;
        delete this.__containerEvents;
    }
    TileEntityBase.prototype.created = function () {
        this.onCreate();
    };
    /** @deprecated */
    TileEntityBase.prototype.init = function () {
        this.region = new WorldRegion(this.blockSource);
        this.onInit();
    };
    /** @deprecated */
    TileEntityBase.prototype.load = function () {
        this.onLoad();
    };
    /** @deprecated */
    TileEntityBase.prototype.unload = function () {
        this.onUnload();
    };
    /** @deprecated */
    TileEntityBase.prototype.tick = function () {
        this.onTick();
    };
    /** @deprecated */
    TileEntityBase.prototype.click = function () { };
    /**
     * Called when a TileEntity is created
     */
    TileEntityBase.prototype.onCreate = function () { };
    /**
     * Called when a TileEntity is initialised in the world
     */
    TileEntityBase.prototype.onInit = function () { };
    /**
     * Called when a chunk with TileEntity is loaded
     */
    TileEntityBase.prototype.onLoad = function () { };
    /**
     * Called when a chunk with TileEntity is unloaded
     */
    TileEntityBase.prototype.onUnload = function () { };
    /**
     * Called every tick and should be used for all the updates of the TileEntity
     */
    TileEntityBase.prototype.onTick = function () { };
    /**
     * Called when the client copy is created
     */
    TileEntityBase.prototype.clientLoad = function () { };
    /**
     * Called on destroying the client copy
     */
    TileEntityBase.prototype.clientUnload = function () { };
    /**
     * Called every tick on client thread
     */
    TileEntityBase.prototype.clientTick = function () { };
    TileEntityBase.prototype.onCheckerTick = function (isInitialized, isLoaded, wasLoaded) { };
    TileEntityBase.prototype.getScreenName = function (player, coords) {
        return "main";
    };
    TileEntityBase.prototype.getScreenByName = function (screenName, container) {
        return null;
    };
    /** @deprecated */
    TileEntityBase.prototype.getGuiScreen = function () {
        return null;
    };
    /**
     * Called when player uses some item on a TileEntity. Replaces "click" function.
     * @returns true if should prevent opening UI.
    */
    TileEntityBase.prototype.onItemUse = function (coords, item, player) {
        return false;
    };
    /**
     * Prevents all actions on click
     */
    TileEntityBase.prototype.preventClick = function () {
        this._clickPrevented = true;
    };
    TileEntityBase.prototype.onItemClick = function (id, count, data, coords, player, extra) {
        if (!this.__initialized) {
            if (!this._runInit()) {
                return false;
            }
        }
        this._clickPrevented = false;
        if (this.onItemUse(coords, new ItemStack(id, count, data, extra), player) || this._clickPrevented) {
            return this._clickPrevented;
        }
        if (Entity.getSneaking(player)) {
            return false;
        }
        var screenName = this.getScreenName(player, coords);
        if (screenName && this.getScreenByName(screenName, this.container)) {
            var client = Network.getClientForPlayer(player);
            if (client) {
                this.container.openFor(client, screenName);
                return true;
            }
        }
        return false;
    };
    TileEntityBase.prototype.destroyBlock = function (coords, player) { };
    /** @deprecated */
    TileEntityBase.prototype.redstone = function (params) {
        this.onRedstoneUpdate(params.power);
    };
    /**
     * Occurs when redstone signal on TileEntity block was updated
     * @param signal signal power (0-15)
     */
    TileEntityBase.prototype.onRedstoneUpdate = function (signal) { };
    TileEntityBase.prototype.projectileHit = function (coords, target) { };
    TileEntityBase.prototype.destroy = function () {
        return false;
    };
    TileEntityBase.prototype.selfDestroy = function () {
        TileEntity.destroyTileEntity(this);
    };
    TileEntityBase.prototype.requireMoreLiquid = function (liquid, amount) { };
    TileEntityBase.prototype.updateLiquidScale = function (scale, liquid) {
        this.container.sendEvent("setLiquidScale", { scale: scale, liquid: liquid, amount: this.liquidStorage.getRelativeAmount(liquid) });
    };
    TileEntityBase.prototype.setLiquidScale = function (container, window, content, data) {
        var gui = container.getUiAdapter();
        if (gui) {
            var size = gui.getBinding(data.scale, "element_rect");
            if (!size)
                return;
            var texture = LiquidRegistry.getLiquidUITexture(data.liquid, size.width(), size.height());
            gui.setBinding(data.scale, "texture", texture);
            gui.setBinding(data.scale, "value", data.amount);
        }
    };
    __decorate([
        BlockEngine.Decorators.ContainerEvent(Side.Client)
    ], TileEntityBase.prototype, "setLiquidScale", null);
    return TileEntityBase;
}());
/// <reference path="./item/interfaces/LiquidItem.ts" />
/**
 * Registry for liquid storage items. Compatible with LiquidRegistry and extends it
 * by adding items that can contain partial amounts of liquid.
 */
var LiquidItemRegistry;
(function (LiquidItemRegistry) {
    ;
    ;
    LiquidItemRegistry.EmptyByFull = {};
    LiquidItemRegistry.FullByEmpty = {};
    LiquidItemRegistry.LiquidItems = {};
    function getEmptyByFullMapping(id, data) {
        return LiquidItemRegistry.EmptyByFull["".concat(id, ":").concat(data)] || LiquidItemRegistry.EmptyByFull["".concat(id, ":-1")];
    }
    function getFullByEmptyMapping(id, data, liquid) {
        return LiquidItemRegistry.FullByEmpty["".concat(id, ":").concat(data, ":").concat(liquid)] || LiquidItemRegistry.FullByEmpty["".concat(id, ":-1:").concat(liquid)];
    }
    function registerItem(liquid, empty, full, amount) {
        if (typeof empty == "number") { // reverse compatibility
            return registerItem(liquid, { id: empty, data: 0 }, { id: full, data: 0 }, amount);
        }
        LiquidItemRegistry.EmptyByFull[full.id + ':' + full.data] = { id: empty.id, data: empty.data == -1 ? 0 : empty.data, liquid: liquid, amount: amount };
        LiquidItemRegistry.FullByEmpty[empty.id + ':' + empty.data + ':' + liquid] = { id: full.id, data: full.data == -1 ? 0 : full.data, amount: amount };
        if (amount == 1000)
            LiquidRegistry.registerItem(liquid, empty, full);
    }
    LiquidItemRegistry.registerItem = registerItem;
    /**
     * Registers item with abstract interface to work with item liquid storage
     * @param itemId item numeric id
     * @param interface liquid item interface object
     */
    function registerItemInterface(itemId, interface) {
        LiquidItemRegistry.LiquidItems[itemId] = interface;
    }
    LiquidItemRegistry.registerItemInterface = registerItemInterface;
    /**
     * Returns liquid item interface for the specified item id
     * @param itemId item numeric id
     */
    function getItemInterface(itemId) {
        return LiquidItemRegistry.LiquidItems[itemId] || null;
    }
    LiquidItemRegistry.getItemInterface = getItemInterface;
    function getItemLiquid(id, data, extra) {
        var liquidItem = LiquidItemRegistry.LiquidItems[id];
        if (liquidItem) {
            return liquidItem.getLiquidStored(data, extra);
        }
        var empty = getEmptyByFullMapping(id, data);
        if (empty) {
            return empty.liquid;
        }
        return LiquidRegistry.getItemLiquid(id, data);
    }
    LiquidItemRegistry.getItemLiquid = getItemLiquid;
    function canBeFilledWithLiquid(id, data, extra, liquid) {
        var liquidItem = LiquidItemRegistry.LiquidItems[id];
        if (liquidItem) {
            var liquidStored = liquidItem.getLiquidStored(data, extra);
            return !liquidStored && liquidItem.isValidLiquid(liquid) ||
                liquidStored == liquid && liquidItem.getAmount(data, extra) < liquidItem.liquidStorage;
        }
        return !!getFullByEmptyMapping(id, data, liquid) || !!LiquidRegistry.getFullItem(id, data, liquid);
    }
    LiquidItemRegistry.canBeFilledWithLiquid = canBeFilledWithLiquid;
    /** @deprecated */
    function getEmptyItem(id, data) {
        var emptyData = getEmptyByFullMapping(id, data);
        if (emptyData) {
            return { id: emptyData.id, count: 1, data: emptyData.data, extra: null, liquid: emptyData.liquid, amount: emptyData.amount };
        }
        var externalEmpty = LiquidRegistry.getEmptyItem(id, data);
        if (externalEmpty) {
            return { id: externalEmpty.id, count: 1, data: externalEmpty.data, extra: null, liquid: externalEmpty.liquid, amount: 1000 };
        }
        return null;
    }
    LiquidItemRegistry.getEmptyItem = getEmptyItem;
    /** @deprecated */
    function getFullItem(id, data, liquid) {
        var fullData = getFullByEmptyMapping(id, data, liquid);
        if (fullData) {
            return { id: fullData.id, count: 1, data: fullData.data, extra: null, amount: fullData.amount };
        }
        var externalFull = LiquidRegistry.getFullItem(id, data, liquid);
        if (externalFull) {
            return { id: externalFull.id, count: 1, data: externalFull.data, extra: null, amount: 1000 };
        }
        return null;
    }
    LiquidItemRegistry.getFullItem = getFullItem;
    function getEmptyStackInternal(id, data, extra) {
        var liquidItem = LiquidItemRegistry.LiquidItems[id];
        if (liquidItem) {
            var amount = liquidItem.getAmount(data, extra);
            if (amount == 0)
                return null;
            var emptyItem = liquidItem.getEmptyItem();
            return { id: emptyItem.id, count: 1, data: emptyItem.data, extra: emptyItem.extra || null, liquid: liquidItem.getLiquidStored(data, extra), amount: amount };
        }
        return getEmptyItem(id, data);
    }
    function getFullStackInternal(id, data, extra, liquid) {
        var liquidItem = LiquidItemRegistry.LiquidItems[id];
        if (liquidItem && liquidItem.isValidLiquid(liquid)) {
            var liquidStored = liquidItem.getLiquidStored(data, extra);
            if (liquidStored && liquidStored != liquid)
                return null;
            var fullItem = liquidItem.getFullItem(liquid);
            if (!fullItem)
                return null;
            var freeAmount = liquidItem.liquidStorage - liquidItem.getAmount(data, extra);
            if (freeAmount == 0)
                return null;
            return { id: fullItem.id, count: 1, data: fullItem.data, extra: fullItem.extra || null, amount: freeAmount };
        }
        return getFullItem(id, data, liquid);
    }
    function getEmptyStack(id, data, extra) {
        if (typeof id == "number") {
            return getEmptyStackInternal(id, data, extra);
        }
        var item = id;
        return getEmptyStackInternal(item.id, item.data, item.extra);
    }
    LiquidItemRegistry.getEmptyStack = getEmptyStack;
    function getFullStack(id, data, extra, liquid) {
        if (typeof id == "number") {
            return getFullStackInternal(id, data, extra, liquid);
        }
        var item = id;
        return getFullStackInternal(item.id, item.data, item.extra, data);
    }
    LiquidItemRegistry.getFullStack = getFullStack;
    registerItem("water", { id: VanillaItemID.glass_bottle, data: 1 }, { id: VanillaItemID.potion, data: 0 }, 250);
})(LiquidItemRegistry || (LiquidItemRegistry = {}));
var BlockEngine;
(function (BlockEngine) {
    /**
     * Class to store and manipulate liquids in TileEntity.
     */
    var LiquidTank = /** @class */ (function () {
        /**
         * Creates new instance of `LiquidTank` and binds it to TileEntity.
         * @param tileEntity TileEntity instance
         * @param name liquid tank name
         * @param limit max liquid amount
         * @param liquids types of valid liquids
         */
        function LiquidTank(tileEntity, name, limit, liquids) {
            this.name = name;
            this.limit = limit;
            if (liquids)
                this.setValidLiquids(liquids);
            this.setParent(tileEntity);
        }
        /**
         * Binds liquid tank to TileEntity.
         * @param tileEntity TileEntity instance
         */
        LiquidTank.prototype.setParent = function (tileEntity) {
            this.tileEntity = tileEntity;
            var liquidData = tileEntity.data[this.name] || {
                liquid: null,
                amount: 0
            };
            tileEntity.data[this.name] = this.data = liquidData;
        };
        /**
         * Gets type of liquid stored in tank.
         * @returns liquid type
         */
        LiquidTank.prototype.getLiquidStored = function () {
            return this.data.liquid;
        };
        /**
         * Gets max amount of liquid in tank.
         * @returns amount of liquid
         */
        LiquidTank.prototype.getLimit = function () {
            return this.limit;
        };
        /**
         * @param liquid liquid type
         * @returns true if liquid can be stored in tank, false otherwise.
         */
        LiquidTank.prototype.isValidLiquid = function (liquid) {
            if (!this.liquids) {
                return true;
            }
            return this.liquids[liquid] || false;
        };
        /**
         * Sets liquids that can be stored in tank.
         * @param liquids arrays of liquid types
         */
        LiquidTank.prototype.setValidLiquids = function (liquids) {
            this.liquids = {};
            for (var _i = 0, liquids_1 = liquids; _i < liquids_1.length; _i++) {
                var name = liquids_1[_i];
                this.liquids[name] = true;
            }
        };
        /**
         * Gets amount of liquid in tank. If `liquid` parameter is set,
         * returns amount of the specified liquid.
         * @param liquid liquid type
         * @returns amount of liquid
         */
        LiquidTank.prototype.getAmount = function (liquid) {
            if (!liquid || this.data.liquid == liquid) {
                return this.data.amount;
            }
            return 0;
        };
        /**
         * Sets liquid to tank.
         * @param liquid liquid type
         * @param amount amount of liquid
         */
        LiquidTank.prototype.setAmount = function (liquid, amount) {
            this.data.liquid = liquid;
            this.data.amount = amount;
        };
        /**
         * Gets amount of liquid divided by max amount.
         * @returns scalar value from 0 to 1
         */
        LiquidTank.prototype.getRelativeAmount = function () {
            return this.data.amount / this.limit;
        };
        /**
         * Adds liquid to tank.
         * @param liquid liquid type
         * @param amount amount of liquid to add
         * @returns amount of liquid that wasn't added
         */
        LiquidTank.prototype.addLiquid = function (liquid, amount) {
            if (!this.data.liquid || this.data.liquid == liquid) {
                this.data.liquid = liquid;
                var add = Math.min(amount, this.limit - this.data.amount);
                this.data.amount += add;
                return amount - add;
            }
            return 0;
        };
        LiquidTank.prototype.getLiquid = function (liquid, amount) {
            if (typeof liquid == "number") {
                amount = liquid;
                liquid = null;
            }
            if (!liquid || this.data.liquid == liquid) {
                var got = Math.min(amount, this.data.amount);
                this.data.amount -= got;
                if (this.data.amount == 0) {
                    this.data.liquid = null;
                }
                return got;
            }
            return 0;
        };
        /**
         * @returns true if tank is full, false otherwise
         */
        LiquidTank.prototype.isFull = function () {
            return this.data.amount >= this.limit;
        };
        /**
         * @returns true if tank is empty, false otherwise
         */
        LiquidTank.prototype.isEmpty = function () {
            return this.data.amount <= 0;
        };
        /**
         * Tries to fill item with liquid from tank.
         * @param inputSlot slot for empty item
         * @param outputSlot slot for full item
         * @returns true if liquid was added, false otherwise.
         */
        LiquidTank.prototype.addLiquidToItem = function (inputSlot, outputSlot) {
            var liquid = this.getLiquidStored();
            if (!liquid)
                return false;
            var amount = this.getAmount(liquid);
            if (amount > 0) {
                var fullStack = LiquidItemRegistry.getFullStack(inputSlot, liquid);
                if (fullStack && this.canStackBeMerged(fullStack, outputSlot)) {
                    if (amount >= fullStack.amount) {
                        this.getLiquid(fullStack.amount);
                        inputSlot.setSlot(inputSlot.id, inputSlot.count - 1, inputSlot.data, inputSlot.extra);
                        inputSlot.validate();
                        outputSlot.setSlot(fullStack.id, outputSlot.count + 1, fullStack.data, fullStack.extra);
                        return true;
                    }
                    var liquidItem = LiquidItemRegistry.getItemInterface(fullStack.id);
                    if (liquidItem && inputSlot.count == 1) {
                        var addedAmount = liquidItem.addLiquid(inputSlot, liquid, amount);
                        this.getLiquid(addedAmount);
                        inputSlot.markDirty();
                        return true;
                    }
                }
            }
            return false;
        };
        /**
         * Tries to fill tank with liquid from item.
         * @param inputSlot slot for full item
         * @param outputSlot slot for empty item
         * @returns true if liquid was extracted, false otherwise.
         */
        LiquidTank.prototype.getLiquidFromItem = function (inputSlot, outputSlot) {
            var liquid = this.getLiquidStored();
            var emptyStack = LiquidItemRegistry.getEmptyStack(inputSlot);
            if (emptyStack && (!liquid && this.isValidLiquid(emptyStack.liquid) || emptyStack.liquid == liquid) && !this.isFull() && this.canStackBeMerged(emptyStack, outputSlot)) {
                var freeAmount = this.getLimit() - this.getAmount();
                if (freeAmount >= emptyStack.amount) {
                    this.addLiquid(emptyStack.liquid, emptyStack.amount);
                    inputSlot.setSlot(inputSlot.id, inputSlot.count - 1, inputSlot.data, inputSlot.extra);
                    inputSlot.validate();
                    outputSlot.setSlot(emptyStack.id, outputSlot.count + 1, emptyStack.data, emptyStack.extra);
                    return true;
                }
                var liquidItem = LiquidItemRegistry.getItemInterface(inputSlot.id);
                if (liquidItem && inputSlot.count == 1) {
                    var extractedAmount = liquidItem.getLiquid(inputSlot, freeAmount);
                    this.addLiquid(emptyStack.liquid, extractedAmount);
                    inputSlot.markDirty();
                    return true;
                }
            }
            return false;
        };
        /**
         * Updates UI bar of liquid. Uses LiquidStorage method for legacy container
         * and container event from TileEntityBase for multiplayer container.
         * @param scale name of liquid bar
         */
        LiquidTank.prototype.updateUiScale = function (scale) {
            var container = this.tileEntity.container;
            if (container.isLegacyContainer()) {
                this.tileEntity.liquidStorage._setContainerScale(container, scale, this.data.liquid, this.getRelativeAmount());
            }
            else {
                container.sendEvent("setLiquidScale", { scale: scale, liquid: this.data.liquid, amount: this.getRelativeAmount() });
            }
        };
        LiquidTank.prototype.canStackBeMerged = function (inputStack, outputStack) {
            return outputStack.id == 0 || (outputStack.id == inputStack.id && outputStack.data == inputStack.data &&
                outputStack.count + inputStack.count <= Item.getMaxStack(outputStack.id, outputStack.data) &&
                outputStack.extra == inputStack.extra);
        };
        return LiquidTank;
    }());
    BlockEngine.LiquidTank = LiquidTank;
})(BlockEngine || (BlockEngine = {}));
// classes
EXPORT("ItemStack", ItemStack);
EXPORT("Vector3", Vector3);
EXPORT("WorldRegion", WorldRegion);
EXPORT("PlayerEntity", PlayerEntity);
EXPORT("BlockBase", BlockBase);
EXPORT("BlockRotative", BlockRotative);
EXPORT("TileEntityBase", TileEntityBase);
EXPORT("ItemCommon", ItemCommon);
EXPORT("ItemFood", ItemFood);
EXPORT("ItemThrowable", ItemThrowable);
EXPORT("ItemArmor", ItemArmor);
EXPORT("ItemTool", ItemTool);
EXPORT("ToolType", ToolType);
// enums
EXPORT("Side", Side);
EXPORT("ItemCategory", ItemCategory);
EXPORT("EnumRarity", EnumRarity);
EXPORT("MiningLevel", MiningLevel);
// APIs
EXPORT("BlockEngine", BlockEngine);
EXPORT("BlockModeler", BlockModeler);
EXPORT("BlockRegistry", BlockRegistry);
EXPORT("ItemRegistry", ItemRegistry);
EXPORT("LiquidItemRegistry", LiquidItemRegistry);
EXPORT("EntityCustomData", EntityCustomData);
EXPORT("IDConverter", IDConverter);
EXPORT("VirtualBlockData", VirtualBlockData);
